<!--

@license Apache-2.0

Copyright (c) 2018 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# Entropy

> [Inverse gamma][invgamma-distribution] distribution [differential entropy][entropy].

<!-- Section to include introductory text. Make sure to keep an empty line after the intro `section` element and another before the `/section` close. -->

<section class="intro">

The [differential entropy][entropy] (in [nats][nats]) for an [inverse gamma][invgamma-distribution] random variable is

<!-- <equation class="equation" label="eq:invgamma_entropy" align="center" raw="h\left( X \right) = \alpha \!+\!\ln(\beta \, \Gamma (\alpha ))\!-\!(1\!+\!\alpha )\Psi (\alpha )" alt="Differential entropy for an inverse gamma distribution."> -->

<div class="equation" align="center" data-raw-text="h\left( X \right) = \alpha \!+\!\ln(\beta \, \Gamma (\alpha ))\!-\!(1\!+\!\alpha )\Psi (\alpha )" data-equation="eq:invgamma_entropy">
    <img src="https://cdn.jsdelivr.net/gh/stdlib-js/stdlib@51534079fef45e990850102147e8945fb023d1d0/lib/node_modules/@stdlib/stats/base/dists/invgamma/entropy/docs/img/equation_invgamma_entropy.svg" alt="Differential entropy for an inverse gamma distribution.">
    <br>
</div>

<!-- </equation> -->

where `α > 0` is the shape parameter, `β > 0` is the rate parameter, `Γ` and denotes the [gamma][gamma-function] and `Ψ` the [digamma][digamma] function.

</section>

<!-- /.intro -->

<!-- Package usage documentation. -->

<section class="usage">

## Usage

```javascript
var entropy = require( '@stdlib/stats/base/dists/invgamma/entropy' );
```

#### entropy( alpha, beta )

Returns the [differential entropy][entropy] of an [inverse gamma][invgamma-distribution] distribution with shape parameter `alpha` and rate parameter `beta` (in [nats][nats]).

```javascript
var v = entropy( 1.0, 1.0 );
// returns ~2.154

v = entropy( 4.0, 12.0 );
// returns ~1.996

v = entropy( 8.0, 2.0 );
// returns ~-0.922
```

If provided `NaN` as any argument, the function returns `NaN`.

```javascript
var v = entropy( NaN, 2.0 );
// returns NaN

v = entropy( 2.0, NaN );
// returns NaN
```

If provided `alpha <= 0`, the function returns `NaN`.

```javascript
var v = entropy( 0.0, 1.0 );
// returns NaN

v = entropy( -1.0, 1.0 );
// returns NaN
```

If provided `beta <= 0`, the function returns `NaN`.

```javascript
var v = entropy( 1.0, 0.0 );
// returns NaN

v = entropy( 1.0, -1.0 );
// returns NaN
```

</section>

<!-- /.usage -->

<!-- Package usage notes. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="notes">

</section>

<!-- /.notes -->

<!-- Package usage examples. -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var randu = require( '@stdlib/random/base/randu' );
var EPS = require( '@stdlib/constants/float64/eps' );
var entropy = require( '@stdlib/stats/base/dists/invgamma/entropy' );

var alpha;
var beta;
var v;
var i;

for ( i = 0; i < 10; i++ ) {
    alpha = ( randu()*10.0 ) + EPS;
    beta = ( randu()*10.0 ) + EPS;
    v = entropy( alpha, beta );
    console.log( 'α: %d, β: %d, h(X;α,β): %d', alpha.toFixed( 4 ), beta.toFixed( 4 ), v.toFixed( 4 ) );
}
```

</section>

<!-- /.examples -->

<!-- Section to include cited references. If references are included, add a horizontal rule *before* the section. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="references">

</section>

<!-- /.references -->

<!-- Section for all links. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="links">

[digamma]: https://en.wikipedia.org/wiki/Digamma_function

[gamma-function]: https://en.wikipedia.org/wiki/Gamma_function

[invgamma-distribution]: https://en.wikipedia.org/wiki/Inverse-gamma_distribution

[entropy]: https://en.wikipedia.org/wiki/Entropy_%28information_theory%29

[nats]: https://en.wikipedia.org/wiki/Nat_%28unit%29

</section>

<!-- /.links -->
