<?php

/***************************************************************************\
 *  SPIP, Système de publication pour l'internet                           *
 *                                                                         *
 *  Copyright © avec tendresse depuis 2001                                 *
 *  Arnaud Martin, Antoine Pitrou, Philippe Rivière, Emmanuel Saint-James  *
 *                                                                         *
 *  Ce programme est un logiciel libre distribué sous licence GNU/GPL.     *
 *  Pour plus de détails voir le fichier COPYING.txt ou l'aide en ligne.   *
\***************************************************************************/

/**
 * Gestion du compage des statistiques de visites (cron)
 *
 * @plugin Statistiques pour SPIP
 * @license GNU/GPL
 * @package SPIP\Statistiques\Genie
 **/

if (!defined('_ECRIRE_INC_VERSION')) {
	return;
}
if (!defined('_CRON_LOT_FICHIERS_VISITE')) {
	define('_CRON_LOT_FICHIERS_VISITE', 100);
}

### Pour se debarrasser du md5, comment faire ? Un index sur 'referer' ?
### ou alors la meme notion, mais sans passer par des fonctions HEX ?


/**
 * Prendre en compte un fichier de visite
 *
 * @param string $fichier
 *     Nom du fichier de visite
 * @param int $visites
 *     Nombre de visites
 * @param array $visites_a
 *     Couples id_article => nombre : comptage par identifiant d'article
 * @param array $referers
 *     Couples url_referer => nombre : comptage par url de referer
 * @param array $referers_a
 *     Couples id_article => array (url_referer => nombre) : comptage par article puis url de referer
 * @return void
 **/
function compte_fichier_visite($fichier, &$visites, &$visites_a, &$referers, &$referers_a) {

	// Noter la visite du site (article 0)
	$visites++;

	$content = [];
	if (lire_fichier($fichier, $content)) {
		$content = @unserialize($content);
	}
	if (!is_array($content)) {
		return;
	}

	foreach ($content as $source => $num) {
		[$log_type, $log_id_num, $log_referer]
			= preg_split(",\t,", $source, 3);

		// Noter le referer
		if ($log_referer) {
			$log_referer = nettoyer_referer($log_referer);
			if (!isset($referers[$log_referer])) {
				$referers[$log_referer] = 0;
			}
			$referers[$log_referer]++;
		}

		// S'il s'agit d'un article, noter ses visites
		if (
			$log_type == 'article'
			and $id_article = intval($log_id_num)
		) {
			if (!isset($visites_a[$id_article])) {
				$visites_a[$id_article] = 0;
			}
			$visites_a[$id_article]++;
			if ($log_referer) {
				if (!isset($referers_a[$id_article][$log_referer])) {
					$referers_a[$id_article][$log_referer] = 0;
				}
				$referers_a[$id_article][$log_referer]++;
			}
		}
	}
}


/**
 * Calcule les statistiques de visites, en plusieurs étapes
 *
 * @uses compte_fichier_visite()
 * @uses genie_popularite_constantes()
 *
 * @param int $t
 *     Timestamp de la dernière exécution de cette tâche
 * @return null|int
 *     - null si aucune visite à prendre en compte ou si tous les fichiers de visite sont traités,
 *     - entier négatif s'il reste encore des fichiers à traiter
 **/
function calculer_visites($t) {
	include_spip('base/abstract_sql');

	// Initialisations
	$visites = []; # visites du site
	$visites_a = []; # tableau des visites des articles
	$referers = []; # referers du site
	$referers_a = []; # tableau des referers des articles

	// charger un certain nombre de fichiers de visites,
	// et faire les calculs correspondants

	// Traiter jusqu'a 100 sessions datant d'au moins 30 minutes
	$sessions = preg_files(sous_repertoire(_DIR_TMP, 'visites'));

	$compteur = _CRON_LOT_FICHIERS_VISITE;
	$date_init = time() - 30 * 60;
	foreach ($sessions as $item) {
		if (($d = @filemtime($item)) < $date_init) {
			if (!$d) {
				$d = $date_init;
			} // si le fs ne donne pas de date, on prend celle du traitement, mais tout cela risque d'etre bien douteux
			$d = date('Y-m-d', $d);
			spip_log("traite la session $item");
			compte_fichier_visite(
				$item,
				$visites[$d],
				$visites_a[$d],
				$referers[$d],
				$referers_a[$d]
			);
			spip_unlink($item);
			if (--$compteur <= 0) {
				break;
			}
		}
		#else spip_log("$item pas vieux");
	}
	if (!count($visites)) {
		return;
	}

	include_spip('genie/popularites');
	[$a, $b] = genie_popularite_constantes(24 * 3600);

	// Maintenant on dispose de plusieurs tableaux qu'il faut ventiler dans
	// les tables spip_visites, spip_visites_articles, spip_referers
	// et spip_referers_articles ; attention a affecter tout ca a la bonne
	// date (celle de la visite, pas celle du traitement)
	foreach (array_keys($visites) as $date) {
		if ($visites[$date]) {
			// 1. les visites du site (facile)
			if (!sql_countsel('spip_visites', "date='$date'")) {
				sql_insertq(
					'spip_visites',
					['date' => $date, 'visites' => $visites[$date]]
				);
			} else {
				sql_update('spip_visites', ['visites' => 'visites+' . intval($visites[$date])], "date='$date'");
			}

			// 2. les visites des articles
			if ($visites_a[$date]) {
				$ar = [];  # tableau num -> liste des articles ayant num visites
				foreach ($visites_a[$date] as $id_article => $n) {
					if (
						!sql_countsel(
							'spip_visites_articles',
							"id_article=$id_article AND date='$date'"
						)
					) {
						sql_insertq(
							'spip_visites_articles',
							[
								'id_article' => $id_article,
								'visites' => 0,
								'date' => $date
							]
						);
					}
					$ar[$n][] = $id_article;
				}
				foreach ($ar as $n => $liste) {
					$tous = sql_in('id_article', $liste);
					sql_update(
						'spip_visites_articles',
						['visites' => "visites+$n"],
						"date='$date' AND $tous"
					);

					$ref = $noref = [];
					foreach ($liste as $id) {
						if (isset($referers_a[$id])) {
							$ref[] = $id;
						} else {
							$noref[] = $id;
						}
					}
					// il faudrait ponderer la popularite ajoutee ($n) par son anciennete eventuelle
					// sur le modele de ce que fait genie/popularites
					if (is_countable($noref) ? count($noref) : 0) {
						sql_update(
							'spip_articles',
							[
								'visites' => "visites+$n",
								'popularite' => 'popularite+' . number_format(round($n * $b, 2), 2, '.', ''),
								'maj' => 'maj'
							],
							sql_in('id_article', $noref)
						);
					}

					if (is_countable($ref) ? count($ref) : 0) {
						sql_update(
							'spip_articles',
							[
								'visites' => 'visites+' . ($n + 1),
								'popularite' => 'popularite+' . number_format(round($n * $b, 2), 2, '.', ''),
								'maj' => 'maj'
							],
							sql_in('id_article', $ref)
						);
					}

					## Ajouter un JOIN sur le statut de l'article ?
				}
			}
			if (!isset($GLOBALS['meta']['activer_referers']) or $GLOBALS['meta']['activer_referers'] == 'oui') {
				// 3. Les referers du site
				// insertion pour les nouveaux, au tableau des increments sinon
				if (isset($referers[$date])) {
					$ar = [];
					$trouver_table = charger_fonction('trouver_table', 'base');
					$desc = $trouver_table('referers');
					$n = preg_match('/(\d+)/', $desc['field']['referer'], $r);
					$n = $n ? $r[1] : 255;
					foreach ($referers[$date] as $referer => $num) {
						$referer_md5 = sql_hex(substr(md5($referer), 0, 15));
						$referer = substr($referer, 0, $n);
						if (!sql_countsel('spip_referers', "referer_md5=$referer_md5")) {
							sql_insertq(
								'spip_referers',
								[
									'visites' => $num,
									'visites_jour' => $num,
									'visites_veille' => 0,
									'date' => $date,
									'referer' => $referer,
									'referer_md5' => $referer_md5
								]
							);
						} else {
							$ar[$num][] = $referer_md5;
						}
					}

					// appliquer les increments sur les anciens
					// attention on appelle sql_in en mode texte et pas array
					// pour ne pas passer sql_quote() sur les '0x1234' de referer_md5, cf #849
					foreach ($ar as $num => $liste) {
						sql_update(
							'spip_referers',
							['visites' => "visites+$num", 'visites_jour' => "visites_jour+$num"],
							sql_in('referer_md5', join(', ', $liste))
						);
					}
				}

				// 4. Les referers d'articles
				if (isset($referers_a[$date])) {
					$ar = [];
					$insert = [];
					// s'assurer d'un slot pour chacun
					foreach ($referers_a[$date] as $id_article => $referers) {
						foreach ($referers as $referer => $num) {
							$referer_md5 = sql_hex(substr(md5($referer), 0, 15));
							$prim = "(id_article=$id_article AND referer_md5=$referer_md5)";
							if (!sql_countsel('spip_referers_articles', $prim)) {
								sql_insertq(
									'spip_referers_articles',
									[
										'visites' => $num,
										'id_article' => $id_article,
										'referer' => $referer,
										'referer_md5' => $referer_md5
									]
								);
							} else {
								$ar[$num][] = $prim;
							}
						}
					}
					// ajouter les visites
					foreach ($ar as $num => $liste) {
						sql_update('spip_referers_articles', ['visites' => "visites+$num"], join(' OR ', $liste));
						## Ajouter un JOIN sur le statut de l'article ?
					}
				}
			}
		}
	}

	// S'il reste des fichiers a manger, le signaler pour reexecution rapide
	if ($compteur == 0) {
		spip_log('il reste des visites a traiter...');

		return -$t;
	}
}

/**
 * Nettoyer les IPs des flooders 24H apres leur dernier passage
 */
function visites_nettoyer_flood() {
	if (is_dir($dir = _DIR_TMP . 'flood/')) {
		include_spip('inc/invalideur');
		if (!defined('_IP_FLOOD_TTL')) {
			define('_IP_FLOOD_TTL', 24 * 3600);
		} // 24H par defaut
		$options = [
			'mtime' => $_SERVER['REQUEST_TIME'] - _IP_FLOOD_TTL,
		];
		purger_repertoire($dir, $options);
	}
}

/**
 * Nettoyer les urls en enlevant les variables de personnalisation marketing, ou variantes Amp
 */

function nettoyer_referer($url) {

	// &utm_xxx=
	$url = preg_replace('`[?&]utm_.*$`', '', $url);

	// &fbclid=
	$url = preg_replace('`[?&]fbclid.*$`', '', $url);

	// &amp=1
	$url = preg_replace('`[?&]amp=1$`', '', $url);

	return $url ;
}

/**
 * Cron de calcul de statistiques des visites
 *
 * Calcule les stats en plusieurs étapes
 *
 * @uses calculer_visites()
 *
 * @param int $t
 *     Timestamp de la dernière exécution de cette tâche
 * @return int
 *     Positif si la tâche a été terminée, négatif pour réexécuter cette tâche
 **/
function genie_visites_dist($t) {
	$encore = calculer_visites($t);

	// Si ce n'est pas fini on redonne la meme date au fichier .lock
	// pour etre prioritaire lors du cron suivant
	if ($encore) {
		return (0 - $t);
	}

	// nettoyer les IP des floodeurs quand on a fini de compter les stats
	visites_nettoyer_flood();

	return 1;
}
