/*
 *  parseFloat() tests.
 */

var WHITESPACE_CODEPOINTS = [
    // from WhiteSpace production
    0x0009,    // <TAB>
    0x000B,    // <VT>
    0x000C,    // <FF>
    0x0020,    // <SP>
    0x00A0,    // <NBSP>
    0xFEFF,    // <BOM>

    // WhiteSpace production also has <USP>, which means any other Unicode
    // space separator (category Zs), which needs to be checked from (up to
    // date) Unicode data.  The WhiteSpace-Z.txt file, created as part of
    // the build, currently contains (duplicates eliminated):

//  0x0020,    // 0020;SPACE;Zs;0;WS;;;;;N;;;;;
//  0x00A0,    // 00A0;NO-BREAK SPACE;Zs;0;CS;<noBreak> 0020;;;;N;NON-BREAKING SPACE;;;;
    0x1680,    // 1680;OGHAM SPACE MARK;Zs;0;WS;;;;;N;;;;;
    0x2000,    // 2000;EN QUAD;Zs;0;WS;2002;;;;N;;;;;
    0x2001,    // 2001;EM QUAD;Zs;0;WS;2003;;;;N;;;;;
    0x2002,    // 2002;EN SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x2003,    // 2003;EM SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x2004,    // 2004;THREE-PER-EM SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x2005,    // 2005;FOUR-PER-EM SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x2006,    // 2006;SIX-PER-EM SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x2007,    // 2007;FIGURE SPACE;Zs;0;WS;<noBreak> 0020;;;;N;;;;;
    0x2008,    // 2008;PUNCTUATION SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x2009,    // 2009;THIN SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x200A,    // 200A;HAIR SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
//  0x2028,    // 2028;LINE SEPARATOR;Zl;0;WS;;;;;N;;;;;
//  0x2029,    // 2029;PARAGRAPH SEPARATOR;Zp;0;B;;;;;N;;;;;
    0x202F,    // 202F;NARROW NO-BREAK SPACE;Zs;0;CS;<noBreak> 0020;;;;N;;;;;
    0x205F,    // 205F;MEDIUM MATHEMATICAL SPACE;Zs;0;WS;<compat> 0020;;;;N;;;;;
    0x3000,    // 3000;IDEOGRAPHIC SPACE;Zs;0;WS;<wide> 0020;;;;N;;;;;

    // from LineTerminator production
    0x000a,    // <LF>
    0x000d,    // <CR>
    0x2028,    // <LS>
    0x2029,    // <PS>
];

// indirect eval -> this is bound to the global object, E5 Section 10.4.2, step 1.a.
var g = (function () { var e = eval; return e('this'); } )();

/*===
basic cases
Infinity number Infinity
+Infinity number Infinity
-Infinity number -Infinity
   Infinity number Infinity
   +Infinity number Infinity
   -Infinity number -Infinity
   Infinityxyz number Infinity
   +Infinityxyz number Infinity
   -Infinityxyz number -Infinity
123. number 123
+123. number 123
-123. number -123
   123. number 123
   +123. number 123
   -123. number -123
   123.xyz number 123
   +123.xyz number 123
   -123.xyz number -123
123.e1 number 1230
+123.e1 number 1230
-123.e1 number -1230
   123.e1 number 1230
   +123.e1 number 1230
   -123.e1 number -1230
   123.e1xyz number 1230
   +123.e1xyz number 1230
   -123.e1xyz number -1230
123.+e1 number 123
+123.+e1 number 123
-123.+e1 number -123
   123.+e1 number 123
   +123.+e1 number 123
   -123.+e1 number -123
   123.+e1xyz number 123
   +123.+e1xyz number 123
   -123.+e1xyz number -123
123.-e1 number 123
+123.-e1 number 123
-123.-e1 number -123
   123.-e1 number 123
   +123.-e1 number 123
   -123.-e1 number -123
   123.-e1xyz number 123
   +123.-e1xyz number 123
   -123.-e1xyz number -123
123.4 number 123.4
+123.4 number 123.4
-123.4 number -123.4
   123.4 number 123.4
   +123.4 number 123.4
   -123.4 number -123.4
   123.4xyz number 123.4
   +123.4xyz number 123.4
   -123.4xyz number -123.4
123.4e1 number 1234
+123.4e1 number 1234
-123.4e1 number -1234
   123.4e1 number 1234
   +123.4e1 number 1234
   -123.4e1 number -1234
   123.4e1xyz number 1234
   +123.4e1xyz number 1234
   -123.4e1xyz number -1234
123.4e+1 number 1234
+123.4e+1 number 1234
-123.4e+1 number -1234
   123.4e+1 number 1234
   +123.4e+1 number 1234
   -123.4e+1 number -1234
   123.4e+1xyz number 1234
   +123.4e+1xyz number 1234
   -123.4e+1xyz number -1234
123.4e-1 number 12.34
+123.4e-1 number 12.34
-123.4e-1 number -12.34
   123.4e-1 number 12.34
   +123.4e-1 number 12.34
   -123.4e-1 number -12.34
   123.4e-1xyz number 12.34
   +123.4e-1xyz number 12.34
   -123.4e-1xyz number -12.34
.123 number 0.123
+.123 number 0.123
-.123 number -0.123
   .123 number 0.123
   +.123 number 0.123
   -.123 number -0.123
   .123xyz number 0.123
   +.123xyz number 0.123
   -.123xyz number -0.123
.123e1 number 1.23
+.123e1 number 1.23
-.123e1 number -1.23
   .123e1 number 1.23
   +.123e1 number 1.23
   -.123e1 number -1.23
   .123e1xyz number 1.23
   +.123e1xyz number 1.23
   -.123e1xyz number -1.23
.123e+1 number 1.23
+.123e+1 number 1.23
-.123e+1 number -1.23
   .123e+1 number 1.23
   +.123e+1 number 1.23
   -.123e+1 number -1.23
   .123e+1xyz number 1.23
   +.123e+1xyz number 1.23
   -.123e+1xyz number -1.23
.123e-1 number 0.0123
+.123e-1 number 0.0123
-.123e-1 number -0.0123
   .123e-1 number 0.0123
   +.123e-1 number 0.0123
   -.123e-1 number -0.0123
   .123e-1xyz number 0.0123
   +.123e-1xyz number 0.0123
   -.123e-1xyz number -0.0123
123 number 123
+123 number 123
-123 number -123
   123 number 123
   +123 number 123
   -123 number -123
   123xyz number 123
   +123xyz number 123
   -123xyz number -123
123e1 number 1230
+123e1 number 1230
-123e1 number -1230
   123e1 number 1230
   +123e1 number 1230
   -123e1 number -1230
   123e1xyz number 1230
   +123e1xyz number 1230
   -123e1xyz number -1230
123e+1 number 1230
+123e+1 number 1230
-123e+1 number -1230
   123e+1 number 1230
   +123e+1 number 1230
   -123e+1 number -1230
   123e+1xyz number 1230
   +123e+1xyz number 1230
   -123e+1xyz number -1230
123e-1 number 12.3
+123e-1 number 12.3
-123e-1 number -12.3
   123e-1 number 12.3
   +123e-1 number 12.3
   -123e-1 number -12.3
   123e-1xyz number 12.3
   +123e-1xyz number 12.3
   -123e-1xyz number -12.3
not-a-number number NaN
+not-a-number number NaN
-not-a-number number NaN
   not-a-number number NaN
   +not-a-number number NaN
   -not-a-number number NaN
   not-a-numberxyz number NaN
   +not-a-numberxyz number NaN
   -not-a-numberxyz number NaN
===*/

/* Test basic grammar cases.  The relevant grammar rule is StrDecimalLiteral.
 */

print('basic cases');

function basicTest() {
    var base_values = [
        'Infinity',

        '123.',
        '123.e1',
        '123.+e1',
        '123.-e1',
        '123.4',
        '123.4e1',
        '123.4e+1',
        '123.4e-1',

        '.123',
        '.123e1',
        '.123e+1',
        '.123e-1',

        '123',
        '123e1',
        '123e+1',
        '123e-1',

        'not-a-number',
    ];
    var tests = [];
    var i;
    var t;

    for (i = 0; i < base_values.length; i++) {
        t = base_values[i];
        tests.push('' + t);
        tests.push('+' + t);
        tests.push('-' + t);
        tests.push('   ' + t);
        tests.push('   +' + t);
        tests.push('   -' + t);
        tests.push('   ' + t + 'xyz');
        tests.push('   +' + t + 'xyz');
        tests.push('   -' + t + 'xyz');
    }

    for (i = 0; i < tests.length; i++) {
        t = g.parseFloat(tests[i]);
        print(tests[i], typeof t, t);
    }
}

try {
    basicTest();
} catch (e) {
    print(e.name);
}

/*===
coercion
toString()
123.4
===*/

/* parseFloat coercion: ToString() followed by string parsing. */

print('coercion');

try {
    print(g.parseFloat(
        { toString: function() { print('toString()'); return '123.4'; },
          valueOf: function() { print('valueOf()'); return 123.4; } }
    ));
} catch (e) {
    print(e.name);
}

/*===
whitespace strip test
321.4
123.4
123.4
===*/

/* White space trimming. */

print('whitespace strip test');

function whiteSpaceStripTest() {
    var t = [];
    var i;
    var txt;

    // a simple test with only ASCII whitespace
    txt = '\u0009\u000b\u000c\u0020\u00a0\u000a\u000d321.4';
    print(g.parseFloat(txt));

    // txt will contain every whitespace codepoint followed by the number
    for (i = 0; i < WHITESPACE_CODEPOINTS.length; i++) {
        t.push(String.fromCharCode(WHITESPACE_CODEPOINTS[i]));
    }
    t.push('123.4');
    txt = t.join('');
    print(g.parseFloat(txt));

    // try with trailing garbage, should be allowed
    print(g.parseFloat(txt + 'xyz'));
}

try {
    whiteSpaceStripTest();
} catch (e) {
    print(e.name);
}

/*===
trailing garbage
123
1230
1230
123.4
123.4
123.4
123.4
===*/

/* Trailing garbage is ignored. */

print('trailing garbage');

function trailingGarbageTest() {
    print(g.parseFloat('123xxx'));
    print(g.parseFloat('123e1'));
    print(g.parseFloat('123e1e2'));  // second 'e' is garbage

    print(g.parseFloat('123.4'));
    print(g.parseFloat('123.4.5'));  // second '.' is garbage

    print(g.parseFloat('123.4\u0000'));
    print(g.parseFloat('123.4\ufedc'));

    // XXX: more cases
}

try {
    trailingGarbageTest();
} catch (e) {
    print(e.name);
}

/*===
infinity
number Infinity
number Infinity
number Infinity
number -Infinity
number NaN
number NaN
===*/

/* 'Infinity' is parsed; case sensitive.  Don't parse 'inf'. */

print('infinity');

function infinityTest() {
    var t;

    t = g.parseFloat('Infinity');
    print(typeof t, t);

    t = g.parseFloat('  InfinityXXX');
    print(typeof t, t);

    t = g.parseFloat('  +InfinityXXX');
    print(typeof t, t);

    t = g.parseFloat('  -InfinityXXX');
    print(typeof t, t);

    t = g.parseFloat('infinity');
    print(typeof t, t);

    t = g.parseFloat('inf');
    print(typeof t, t);
}

try {
    infinityTest();
} catch (e) {
    print(e.name);
}

/* Leading plus/minus sign is accepted. */

/*===
hex/oct
number 0
number 0
number 0
number 0
number 77
number 88
number 99
===*/

/* Hex/oct is not accepted.
 *
 * If a number begins with '0x' or '0X' it gets parsed as a zero.  The 'x'/'X'
 * character will be considered garbage and terminates the number.
 *
 * If a number has leading zeroes, they will be accepted as part of the decimal
 * value.  StrDecimalLiteral (unlike DecimalLiteral) accepts leading zeroes.
 */

print('hex/oct');

function hexOctTest() {
    var t;

    t = g.parseFloat('0x1234');
    print(typeof t, t);

    t = g.parseFloat('0xcafe');
    print(typeof t, t);

    t = g.parseFloat('0X1234');
    print(typeof t, t);

    t = g.parseFloat('0Xcafe');
    print(typeof t, t);

    t = g.parseFloat('077');
    print(typeof t, t);

    t = g.parseFloat('088');
    print(typeof t, t);

    t = g.parseFloat('099');
    print(typeof t, t);
}

try {
    hexOctTest();
} catch (e) {
    print(e.name);
}

/*===
exactness
0 1 1 true
1 2 2 true
2 4 4 true
3 8 8 true
4 16 16 true
5 32 32 true
6 64 64 true
7 128 128 true
8 256 256 true
9 512 512 true
10 1024 1024 true
11 2048 2048 true
12 4096 4096 true
13 8192 8192 true
14 16384 16384 true
15 32768 32768 true
16 65536 65536 true
17 131072 131072 true
18 262144 262144 true
19 524288 524288 true
20 1048576 1048576 true
21 2097152 2097152 true
22 4194304 4194304 true
23 8388608 8388608 true
24 16777216 16777216 true
25 33554432 33554432 true
26 67108864 67108864 true
27 134217728 134217728 true
28 268435456 268435456 true
29 536870912 536870912 true
30 1073741824 1073741824 true
31 2147483648 2147483648 true
32 4294967296 4294967296 true
33 8589934592 8589934592 true
34 17179869184 17179869184 true
35 34359738368 34359738368 true
36 68719476736 68719476736 true
37 137438953472 137438953472 true
38 274877906944 274877906944 true
39 549755813888 549755813888 true
40 1099511627776 1099511627776 true
41 2199023255552 2199023255552 true
42 4398046511104 4398046511104 true
43 8796093022208 8796093022208 true
44 17592186044416 17592186044416 true
45 35184372088832 35184372088832 true
46 70368744177664 70368744177664 true
47 140737488355328 140737488355328 true
48 281474976710656 281474976710656 true
49 562949953421312 562949953421312 true
50 1125899906842624 1125899906842624 true
51 2251799813685248 2251799813685248 true
52 4503599627370496 4503599627370496 true
53 9007199254740992 9007199254740992 true
54 18014398509481984 18014398509481984 true
55 36028797018963970 36028797018963970 true
56 72057594037927940 72057594037927940 true
57 144115188075855870 144115188075855870 true
58 288230376151711740 288230376151711740 true
59 576460752303423500 576460752303423500 true
60 1152921504606847000 1152921504606847000 true
61 2305843009213694000 2305843009213694000 true
62 4611686018427388000 4611686018427388000 true
63 9223372036854776000 9223372036854776000 true
64 18446744073709552000 18446744073709552000 true
65 36893488147419103000 36893488147419103000 true
66 73786976294838210000 73786976294838210000 true
67 147573952589676410000 147573952589676410000 true
68 295147905179352830000 295147905179352830000 true
69 590295810358705700000 590295810358705700000 true
70 1.1805916207174113e+21 1.1805916207174113e+21 true
71 2.3611832414348226e+21 2.3611832414348226e+21 true
72 4.722366482869645e+21 4.722366482869645e+21 true
73 9.44473296573929e+21 9.44473296573929e+21 true
74 1.888946593147858e+22 1.888946593147858e+22 true
75 3.777893186295716e+22 3.777893186295716e+22 true
76 7.555786372591432e+22 7.555786372591432e+22 true
77 1.5111572745182865e+23 1.5111572745182865e+23 true
78 3.022314549036573e+23 3.022314549036573e+23 true
79 6.044629098073146e+23 6.044629098073146e+23 true
80 1.2089258196146292e+24 1.2089258196146292e+24 true
81 2.4178516392292583e+24 2.4178516392292583e+24 true
82 4.835703278458517e+24 4.835703278458517e+24 true
83 9.671406556917033e+24 9.671406556917033e+24 true
84 1.9342813113834067e+25 1.9342813113834067e+25 true
85 3.8685626227668134e+25 3.8685626227668134e+25 true
86 7.737125245533627e+25 7.737125245533627e+25 true
87 1.5474250491067253e+26 1.5474250491067253e+26 true
88 3.094850098213451e+26 3.094850098213451e+26 true
89 6.189700196426902e+26 6.189700196426902e+26 true
90 1.2379400392853803e+27 1.2379400392853803e+27 true
91 2.4758800785707605e+27 2.4758800785707605e+27 true
92 4.951760157141521e+27 4.951760157141521e+27 true
93 9.903520314283042e+27 9.903520314283042e+27 true
94 1.9807040628566084e+28 1.9807040628566084e+28 true
95 3.961408125713217e+28 3.961408125713217e+28 true
96 7.922816251426434e+28 7.922816251426434e+28 true
97 1.5845632502852868e+29 1.5845632502852868e+29 true
98 3.1691265005705735e+29 3.1691265005705735e+29 true
99 6.338253001141147e+29 6.338253001141147e+29 true
100 1.2676506002282294e+30 1.2676506002282294e+30 true
101 2.535301200456459e+30 2.535301200456459e+30 true
102 5.070602400912918e+30 5.070602400912918e+30 true
103 1.0141204801825835e+31 1.0141204801825835e+31 true
104 2.028240960365167e+31 2.028240960365167e+31 true
105 4.056481920730334e+31 4.056481920730334e+31 true
106 8.112963841460668e+31 8.112963841460668e+31 true
107 1.6225927682921336e+32 1.6225927682921336e+32 true
108 3.2451855365842673e+32 3.2451855365842673e+32 true
109 6.490371073168535e+32 6.490371073168535e+32 true
110 1.298074214633707e+33 1.298074214633707e+33 true
111 2.596148429267414e+33 2.596148429267414e+33 true
112 5.192296858534828e+33 5.192296858534828e+33 true
113 1.0384593717069655e+34 1.0384593717069655e+34 true
114 2.076918743413931e+34 2.076918743413931e+34 true
115 4.153837486827862e+34 4.153837486827862e+34 true
116 8.307674973655724e+34 8.307674973655724e+34 true
117 1.661534994731145e+35 1.661534994731145e+35 true
118 3.32306998946229e+35 3.32306998946229e+35 true
119 6.64613997892458e+35 6.64613997892458e+35 true
120 1.329227995784916e+36 1.329227995784916e+36 true
121 2.658455991569832e+36 2.658455991569832e+36 true
122 5.316911983139664e+36 5.316911983139664e+36 true
123 1.0633823966279327e+37 1.0633823966279327e+37 true
124 2.1267647932558654e+37 2.1267647932558654e+37 true
125 4.253529586511731e+37 4.253529586511731e+37 true
126 8.507059173023462e+37 8.507059173023462e+37 true
127 1.7014118346046923e+38 1.7014118346046923e+38 true
128 3.402823669209385e+38 3.402823669209385e+38 true
129 6.80564733841877e+38 6.80564733841877e+38 true
130 1.361129467683754e+39 1.361129467683754e+39 true
131 2.722258935367508e+39 2.722258935367508e+39 true
132 5.444517870735016e+39 5.444517870735016e+39 true
133 1.0889035741470031e+40 1.0889035741470031e+40 true
134 2.1778071482940062e+40 2.1778071482940062e+40 true
135 4.3556142965880123e+40 4.3556142965880123e+40 true
136 8.711228593176025e+40 8.711228593176025e+40 true
137 1.742245718635205e+41 1.742245718635205e+41 true
138 3.48449143727041e+41 3.48449143727041e+41 true
139 6.96898287454082e+41 6.96898287454082e+41 true
140 1.393796574908164e+42 1.393796574908164e+42 true
141 2.787593149816328e+42 2.787593149816328e+42 true
142 5.575186299632656e+42 5.575186299632656e+42 true
143 1.1150372599265312e+43 1.1150372599265312e+43 true
144 2.2300745198530623e+43 2.2300745198530623e+43 true
145 4.460149039706125e+43 4.460149039706125e+43 true
146 8.92029807941225e+43 8.92029807941225e+43 true
147 1.78405961588245e+44 1.78405961588245e+44 true
148 3.5681192317649e+44 3.5681192317649e+44 true
149 7.1362384635298e+44 7.1362384635298e+44 true
150 1.42724769270596e+45 1.42724769270596e+45 true
151 2.85449538541192e+45 2.85449538541192e+45 true
152 5.70899077082384e+45 5.70899077082384e+45 true
153 1.141798154164768e+46 1.141798154164768e+46 true
154 2.283596308329536e+46 2.283596308329536e+46 true
155 4.567192616659072e+46 4.567192616659072e+46 true
156 9.134385233318143e+46 9.134385233318143e+46 true
157 1.8268770466636286e+47 1.8268770466636286e+47 true
158 3.6537540933272573e+47 3.6537540933272573e+47 true
159 7.307508186654515e+47 7.307508186654515e+47 true
160 1.461501637330903e+48 1.461501637330903e+48 true
161 2.923003274661806e+48 2.923003274661806e+48 true
162 5.846006549323612e+48 5.846006549323612e+48 true
163 1.1692013098647223e+49 1.1692013098647223e+49 true
164 2.3384026197294447e+49 2.3384026197294447e+49 true
165 4.6768052394588893e+49 4.6768052394588893e+49 true
166 9.353610478917779e+49 9.353610478917779e+49 true
167 1.8707220957835557e+50 1.8707220957835557e+50 true
168 3.7414441915671115e+50 3.7414441915671115e+50 true
169 7.482888383134223e+50 7.482888383134223e+50 true
170 1.4965776766268446e+51 1.4965776766268446e+51 true
171 2.9931553532536892e+51 2.9931553532536892e+51 true
172 5.986310706507379e+51 5.986310706507379e+51 true
173 1.1972621413014757e+52 1.1972621413014757e+52 true
174 2.3945242826029513e+52 2.3945242826029513e+52 true
175 4.789048565205903e+52 4.789048565205903e+52 true
176 9.578097130411805e+52 9.578097130411805e+52 true
177 1.915619426082361e+53 1.915619426082361e+53 true
178 3.831238852164722e+53 3.831238852164722e+53 true
179 7.662477704329444e+53 7.662477704329444e+53 true
180 1.532495540865889e+54 1.532495540865889e+54 true
181 3.064991081731778e+54 3.064991081731778e+54 true
182 6.129982163463556e+54 6.129982163463556e+54 true
183 1.2259964326927111e+55 1.2259964326927111e+55 true
184 2.4519928653854222e+55 2.4519928653854222e+55 true
185 4.9039857307708443e+55 4.9039857307708443e+55 true
186 9.807971461541689e+55 9.807971461541689e+55 true
187 1.9615942923083377e+56 1.9615942923083377e+56 true
188 3.9231885846166755e+56 3.9231885846166755e+56 true
189 7.846377169233351e+56 7.846377169233351e+56 true
190 1.5692754338466702e+57 1.5692754338466702e+57 true
191 3.1385508676933404e+57 3.1385508676933404e+57 true
192 6.277101735386681e+57 6.277101735386681e+57 true
193 1.2554203470773362e+58 1.2554203470773362e+58 true
194 2.5108406941546723e+58 2.5108406941546723e+58 true
195 5.021681388309345e+58 5.021681388309345e+58 true
196 1.004336277661869e+59 1.004336277661869e+59 true
197 2.008672555323738e+59 2.008672555323738e+59 true
198 4.017345110647476e+59 4.017345110647476e+59 true
199 8.034690221294951e+59 8.034690221294951e+59 true
200 1.6069380442589903e+60 1.6069380442589903e+60 true
201 3.2138760885179806e+60 3.2138760885179806e+60 true
202 6.427752177035961e+60 6.427752177035961e+60 true
203 1.2855504354071922e+61 1.2855504354071922e+61 true
204 2.5711008708143844e+61 2.5711008708143844e+61 true
205 5.142201741628769e+61 5.142201741628769e+61 true
206 1.0284403483257538e+62 1.0284403483257538e+62 true
207 2.0568806966515076e+62 2.0568806966515076e+62 true
208 4.113761393303015e+62 4.113761393303015e+62 true
209 8.22752278660603e+62 8.22752278660603e+62 true
210 1.645504557321206e+63 1.645504557321206e+63 true
211 3.291009114642412e+63 3.291009114642412e+63 true
212 6.582018229284824e+63 6.582018229284824e+63 true
213 1.3164036458569648e+64 1.3164036458569648e+64 true
214 2.6328072917139297e+64 2.6328072917139297e+64 true
215 5.2656145834278593e+64 5.2656145834278593e+64 true
216 1.0531229166855719e+65 1.0531229166855719e+65 true
217 2.1062458333711437e+65 2.1062458333711437e+65 true
218 4.2124916667422875e+65 4.2124916667422875e+65 true
219 8.424983333484575e+65 8.424983333484575e+65 true
220 1.684996666696915e+66 1.684996666696915e+66 true
221 3.36999333339383e+66 3.36999333339383e+66 true
222 6.73998666678766e+66 6.73998666678766e+66 true
223 1.347997333357532e+67 1.347997333357532e+67 true
224 2.695994666715064e+67 2.695994666715064e+67 true
225 5.391989333430128e+67 5.391989333430128e+67 true
226 1.0783978666860256e+68 1.0783978666860256e+68 true
227 2.1567957333720512e+68 2.1567957333720512e+68 true
228 4.3135914667441024e+68 4.3135914667441024e+68 true
229 8.627182933488205e+68 8.627182933488205e+68 true
230 1.725436586697641e+69 1.725436586697641e+69 true
231 3.450873173395282e+69 3.450873173395282e+69 true
232 6.901746346790564e+69 6.901746346790564e+69 true
233 1.3803492693581128e+70 1.3803492693581128e+70 true
234 2.7606985387162255e+70 2.7606985387162255e+70 true
235 5.521397077432451e+70 5.521397077432451e+70 true
236 1.1042794154864902e+71 1.1042794154864902e+71 true
237 2.2085588309729804e+71 2.2085588309729804e+71 true
238 4.417117661945961e+71 4.417117661945961e+71 true
239 8.834235323891922e+71 8.834235323891922e+71 true
240 1.7668470647783843e+72 1.7668470647783843e+72 true
241 3.533694129556769e+72 3.533694129556769e+72 true
242 7.067388259113537e+72 7.067388259113537e+72 true
243 1.4134776518227075e+73 1.4134776518227075e+73 true
244 2.826955303645415e+73 2.826955303645415e+73 true
245 5.65391060729083e+73 5.65391060729083e+73 true
246 1.130782121458166e+74 1.130782121458166e+74 true
247 2.261564242916332e+74 2.261564242916332e+74 true
248 4.523128485832664e+74 4.523128485832664e+74 true
249 9.046256971665328e+74 9.046256971665328e+74 true
250 1.8092513943330656e+75 1.8092513943330656e+75 true
251 3.618502788666131e+75 3.618502788666131e+75 true
252 7.237005577332262e+75 7.237005577332262e+75 true
253 1.4474011154664524e+76 1.4474011154664524e+76 true
254 2.894802230932905e+76 2.894802230932905e+76 true
255 5.78960446186581e+76 5.78960446186581e+76 true
256 1.157920892373162e+77 1.157920892373162e+77 true
257 2.315841784746324e+77 2.315841784746324e+77 true
258 4.631683569492648e+77 4.631683569492648e+77 true
259 9.263367138985296e+77 9.263367138985296e+77 true
260 1.8526734277970591e+78 1.8526734277970591e+78 true
261 3.7053468555941183e+78 3.7053468555941183e+78 true
262 7.410693711188237e+78 7.410693711188237e+78 true
263 1.4821387422376473e+79 1.4821387422376473e+79 true
264 2.9642774844752946e+79 2.9642774844752946e+79 true
265 5.928554968950589e+79 5.928554968950589e+79 true
266 1.1857109937901178e+80 1.1857109937901178e+80 true
267 2.3714219875802357e+80 2.3714219875802357e+80 true
268 4.7428439751604714e+80 4.7428439751604714e+80 true
269 9.485687950320943e+80 9.485687950320943e+80 true
270 1.8971375900641885e+81 1.8971375900641885e+81 true
271 3.794275180128377e+81 3.794275180128377e+81 true
272 7.588550360256754e+81 7.588550360256754e+81 true
273 1.517710072051351e+82 1.517710072051351e+82 true
274 3.035420144102702e+82 3.035420144102702e+82 true
275 6.070840288205404e+82 6.070840288205404e+82 true
276 1.2141680576410807e+83 1.2141680576410807e+83 true
277 2.4283361152821613e+83 2.4283361152821613e+83 true
278 4.856672230564323e+83 4.856672230564323e+83 true
279 9.713344461128645e+83 9.713344461128645e+83 true
280 1.942668892225729e+84 1.942668892225729e+84 true
281 3.885337784451458e+84 3.885337784451458e+84 true
282 7.770675568902916e+84 7.770675568902916e+84 true
283 1.5541351137805833e+85 1.5541351137805833e+85 true
284 3.1082702275611665e+85 3.1082702275611665e+85 true
285 6.216540455122333e+85 6.216540455122333e+85 true
286 1.2433080910244666e+86 1.2433080910244666e+86 true
287 2.4866161820489332e+86 2.4866161820489332e+86 true
288 4.9732323640978664e+86 4.9732323640978664e+86 true
289 9.946464728195733e+86 9.946464728195733e+86 true
290 1.9892929456391466e+87 1.9892929456391466e+87 true
291 3.978585891278293e+87 3.978585891278293e+87 true
292 7.957171782556586e+87 7.957171782556586e+87 true
293 1.5914343565113173e+88 1.5914343565113173e+88 true
294 3.1828687130226345e+88 3.1828687130226345e+88 true
295 6.365737426045269e+88 6.365737426045269e+88 true
296 1.2731474852090538e+89 1.2731474852090538e+89 true
297 2.5462949704181076e+89 2.5462949704181076e+89 true
298 5.092589940836215e+89 5.092589940836215e+89 true
299 1.018517988167243e+90 1.018517988167243e+90 true
300 2.037035976334486e+90 2.037035976334486e+90 true
301 4.074071952668972e+90 4.074071952668972e+90 true
302 8.148143905337944e+90 8.148143905337944e+90 true
303 1.629628781067589e+91 1.629628781067589e+91 true
304 3.259257562135178e+91 3.259257562135178e+91 true
305 6.518515124270356e+91 6.518515124270356e+91 true
306 1.3037030248540711e+92 1.3037030248540711e+92 true
307 2.6074060497081422e+92 2.6074060497081422e+92 true
308 5.2148120994162844e+92 5.2148120994162844e+92 true
309 1.0429624198832569e+93 1.0429624198832569e+93 true
310 2.0859248397665138e+93 2.0859248397665138e+93 true
311 4.1718496795330275e+93 4.1718496795330275e+93 true
312 8.343699359066055e+93 8.343699359066055e+93 true
313 1.668739871813211e+94 1.668739871813211e+94 true
314 3.337479743626422e+94 3.337479743626422e+94 true
315 6.674959487252844e+94 6.674959487252844e+94 true
316 1.3349918974505688e+95 1.3349918974505688e+95 true
317 2.6699837949011376e+95 2.6699837949011376e+95 true
318 5.339967589802275e+95 5.339967589802275e+95 true
319 1.067993517960455e+96 1.067993517960455e+96 true
320 2.13598703592091e+96 2.13598703592091e+96 true
321 4.27197407184182e+96 4.27197407184182e+96 true
322 8.54394814368364e+96 8.54394814368364e+96 true
323 1.708789628736728e+97 1.708789628736728e+97 true
324 3.417579257473456e+97 3.417579257473456e+97 true
325 6.835158514946912e+97 6.835158514946912e+97 true
326 1.3670317029893825e+98 1.3670317029893825e+98 true
327 2.734063405978765e+98 2.734063405978765e+98 true
328 5.46812681195753e+98 5.46812681195753e+98 true
329 1.093625362391506e+99 1.093625362391506e+99 true
330 2.187250724783012e+99 2.187250724783012e+99 true
331 4.374501449566024e+99 4.374501449566024e+99 true
332 8.749002899132048e+99 8.749002899132048e+99 true
333 1.7498005798264095e+100 1.7498005798264095e+100 true
334 3.499601159652819e+100 3.499601159652819e+100 true
335 6.999202319305638e+100 6.999202319305638e+100 true
336 1.3998404638611276e+101 1.3998404638611276e+101 true
337 2.7996809277222553e+101 2.7996809277222553e+101 true
338 5.599361855444511e+101 5.599361855444511e+101 true
339 1.1198723710889021e+102 1.1198723710889021e+102 true
340 2.2397447421778042e+102 2.2397447421778042e+102 true
341 4.4794894843556084e+102 4.4794894843556084e+102 true
342 8.958978968711217e+102 8.958978968711217e+102 true
343 1.7917957937422434e+103 1.7917957937422434e+103 true
344 3.583591587484487e+103 3.583591587484487e+103 true
345 7.167183174968974e+103 7.167183174968974e+103 true
346 1.4334366349937947e+104 1.4334366349937947e+104 true
347 2.8668732699875894e+104 2.8668732699875894e+104 true
348 5.733746539975179e+104 5.733746539975179e+104 true
349 1.1467493079950358e+105 1.1467493079950358e+105 true
350 2.2934986159900715e+105 2.2934986159900715e+105 true
351 4.586997231980143e+105 4.586997231980143e+105 true
352 9.173994463960286e+105 9.173994463960286e+105 true
353 1.8347988927920572e+106 1.8347988927920572e+106 true
354 3.6695977855841144e+106 3.6695977855841144e+106 true
355 7.339195571168229e+106 7.339195571168229e+106 true
356 1.4678391142336458e+107 1.4678391142336458e+107 true
357 2.9356782284672915e+107 2.9356782284672915e+107 true
358 5.871356456934583e+107 5.871356456934583e+107 true
359 1.1742712913869166e+108 1.1742712913869166e+108 true
360 2.3485425827738332e+108 2.3485425827738332e+108 true
361 4.6970851655476665e+108 4.6970851655476665e+108 true
362 9.394170331095333e+108 9.394170331095333e+108 true
363 1.8788340662190666e+109 1.8788340662190666e+109 true
364 3.757668132438133e+109 3.757668132438133e+109 true
365 7.515336264876266e+109 7.515336264876266e+109 true
366 1.5030672529752533e+110 1.5030672529752533e+110 true
367 3.0061345059505065e+110 3.0061345059505065e+110 true
368 6.012269011901013e+110 6.012269011901013e+110 true
369 1.2024538023802026e+111 1.2024538023802026e+111 true
370 2.4049076047604052e+111 2.4049076047604052e+111 true
371 4.8098152095208105e+111 4.8098152095208105e+111 true
372 9.619630419041621e+111 9.619630419041621e+111 true
373 1.9239260838083242e+112 1.9239260838083242e+112 true
374 3.8478521676166484e+112 3.8478521676166484e+112 true
375 7.695704335233297e+112 7.695704335233297e+112 true
376 1.5391408670466593e+113 1.5391408670466593e+113 true
377 3.078281734093319e+113 3.078281734093319e+113 true
378 6.156563468186638e+113 6.156563468186638e+113 true
379 1.2313126936373275e+114 1.2313126936373275e+114 true
380 2.462625387274655e+114 2.462625387274655e+114 true
381 4.92525077454931e+114 4.92525077454931e+114 true
382 9.85050154909862e+114 9.85050154909862e+114 true
383 1.970100309819724e+115 1.970100309819724e+115 true
384 3.940200619639448e+115 3.940200619639448e+115 true
385 7.880401239278896e+115 7.880401239278896e+115 true
386 1.5760802478557792e+116 1.5760802478557792e+116 true
387 3.1521604957115583e+116 3.1521604957115583e+116 true
388 6.304320991423117e+116 6.304320991423117e+116 true
389 1.2608641982846233e+117 1.2608641982846233e+117 true
390 2.5217283965692467e+117 2.5217283965692467e+117 true
391 5.0434567931384933e+117 5.0434567931384933e+117 true
392 1.0086913586276987e+118 1.0086913586276987e+118 true
393 2.0173827172553973e+118 2.0173827172553973e+118 true
394 4.034765434510795e+118 4.034765434510795e+118 true
395 8.06953086902159e+118 8.06953086902159e+118 true
396 1.613906173804318e+119 1.613906173804318e+119 true
397 3.227812347608636e+119 3.227812347608636e+119 true
398 6.455624695217272e+119 6.455624695217272e+119 true
399 1.2911249390434543e+120 1.2911249390434543e+120 true
400 2.5822498780869086e+120 2.5822498780869086e+120 true
401 5.164499756173817e+120 5.164499756173817e+120 true
402 1.0328999512347634e+121 1.0328999512347634e+121 true
403 2.065799902469527e+121 2.065799902469527e+121 true
404 4.131599804939054e+121 4.131599804939054e+121 true
405 8.263199609878108e+121 8.263199609878108e+121 true
406 1.6526399219756215e+122 1.6526399219756215e+122 true
407 3.305279843951243e+122 3.305279843951243e+122 true
408 6.610559687902486e+122 6.610559687902486e+122 true
409 1.3221119375804972e+123 1.3221119375804972e+123 true
410 2.6442238751609944e+123 2.6442238751609944e+123 true
411 5.288447750321989e+123 5.288447750321989e+123 true
412 1.0576895500643978e+124 1.0576895500643978e+124 true
413 2.1153791001287955e+124 2.1153791001287955e+124 true
414 4.230758200257591e+124 4.230758200257591e+124 true
415 8.461516400515182e+124 8.461516400515182e+124 true
416 1.6923032801030364e+125 1.6923032801030364e+125 true
417 3.384606560206073e+125 3.384606560206073e+125 true
418 6.769213120412146e+125 6.769213120412146e+125 true
419 1.3538426240824291e+126 1.3538426240824291e+126 true
420 2.7076852481648583e+126 2.7076852481648583e+126 true
421 5.415370496329717e+126 5.415370496329717e+126 true
422 1.0830740992659433e+127 1.0830740992659433e+127 true
423 2.1661481985318866e+127 2.1661481985318866e+127 true
424 4.332296397063773e+127 4.332296397063773e+127 true
425 8.664592794127546e+127 8.664592794127546e+127 true
426 1.7329185588255093e+128 1.7329185588255093e+128 true
427 3.4658371176510186e+128 3.4658371176510186e+128 true
428 6.931674235302037e+128 6.931674235302037e+128 true
429 1.3863348470604074e+129 1.3863348470604074e+129 true
430 2.772669694120815e+129 2.772669694120815e+129 true
431 5.54533938824163e+129 5.54533938824163e+129 true
432 1.109067877648326e+130 1.109067877648326e+130 true
433 2.218135755296652e+130 2.218135755296652e+130 true
434 4.436271510593304e+130 4.436271510593304e+130 true
435 8.872543021186608e+130 8.872543021186608e+130 true
436 1.7745086042373215e+131 1.7745086042373215e+131 true
437 3.549017208474643e+131 3.549017208474643e+131 true
438 7.098034416949286e+131 7.098034416949286e+131 true
439 1.4196068833898572e+132 1.4196068833898572e+132 true
440 2.8392137667797144e+132 2.8392137667797144e+132 true
441 5.678427533559429e+132 5.678427533559429e+132 true
442 1.1356855067118858e+133 1.1356855067118858e+133 true
443 2.2713710134237715e+133 2.2713710134237715e+133 true
444 4.542742026847543e+133 4.542742026847543e+133 true
445 9.085484053695086e+133 9.085484053695086e+133 true
446 1.8170968107390172e+134 1.8170968107390172e+134 true
447 3.6341936214780345e+134 3.6341936214780345e+134 true
448 7.268387242956069e+134 7.268387242956069e+134 true
449 1.4536774485912138e+135 1.4536774485912138e+135 true
450 2.9073548971824276e+135 2.9073548971824276e+135 true
451 5.814709794364855e+135 5.814709794364855e+135 true
452 1.162941958872971e+136 1.162941958872971e+136 true
453 2.325883917745942e+136 2.325883917745942e+136 true
454 4.651767835491884e+136 4.651767835491884e+136 true
455 9.303535670983768e+136 9.303535670983768e+136 true
456 1.8607071341967536e+137 1.8607071341967536e+137 true
457 3.7214142683935073e+137 3.7214142683935073e+137 true
458 7.442828536787015e+137 7.442828536787015e+137 true
459 1.488565707357403e+138 1.488565707357403e+138 true
460 2.977131414714806e+138 2.977131414714806e+138 true
461 5.954262829429612e+138 5.954262829429612e+138 true
462 1.1908525658859223e+139 1.1908525658859223e+139 true
463 2.3817051317718447e+139 2.3817051317718447e+139 true
464 4.7634102635436893e+139 4.7634102635436893e+139 true
465 9.526820527087379e+139 9.526820527087379e+139 true
466 1.9053641054174757e+140 1.9053641054174757e+140 true
467 3.8107282108349515e+140 3.8107282108349515e+140 true
468 7.621456421669903e+140 7.621456421669903e+140 true
469 1.5242912843339806e+141 1.5242912843339806e+141 true
470 3.048582568667961e+141 3.048582568667961e+141 true
471 6.097165137335922e+141 6.097165137335922e+141 true
472 1.2194330274671845e+142 1.2194330274671845e+142 true
473 2.438866054934369e+142 2.438866054934369e+142 true
474 4.877732109868738e+142 4.877732109868738e+142 true
475 9.755464219737476e+142 9.755464219737476e+142 true
476 1.9510928439474951e+143 1.9510928439474951e+143 true
477 3.9021856878949903e+143 3.9021856878949903e+143 true
478 7.804371375789981e+143 7.804371375789981e+143 true
479 1.5608742751579961e+144 1.5608742751579961e+144 true
480 3.1217485503159922e+144 3.1217485503159922e+144 true
481 6.243497100631985e+144 6.243497100631985e+144 true
482 1.248699420126397e+145 1.248699420126397e+145 true
483 2.497398840252794e+145 2.497398840252794e+145 true
484 4.994797680505588e+145 4.994797680505588e+145 true
485 9.989595361011175e+145 9.989595361011175e+145 true
486 1.997919072202235e+146 1.997919072202235e+146 true
487 3.99583814440447e+146 3.99583814440447e+146 true
488 7.99167628880894e+146 7.99167628880894e+146 true
489 1.598335257761788e+147 1.598335257761788e+147 true
490 3.196670515523576e+147 3.196670515523576e+147 true
491 6.393341031047152e+147 6.393341031047152e+147 true
492 1.2786682062094304e+148 1.2786682062094304e+148 true
493 2.557336412418861e+148 2.557336412418861e+148 true
494 5.114672824837722e+148 5.114672824837722e+148 true
495 1.0229345649675443e+149 1.0229345649675443e+149 true
496 2.0458691299350887e+149 2.0458691299350887e+149 true
497 4.0917382598701773e+149 4.0917382598701773e+149 true
498 8.183476519740355e+149 8.183476519740355e+149 true
499 1.636695303948071e+150 1.636695303948071e+150 true
500 3.273390607896142e+150 3.273390607896142e+150 true
501 6.546781215792284e+150 6.546781215792284e+150 true
502 1.3093562431584567e+151 1.3093562431584567e+151 true
503 2.6187124863169135e+151 2.6187124863169135e+151 true
504 5.237424972633827e+151 5.237424972633827e+151 true
505 1.0474849945267654e+152 1.0474849945267654e+152 true
506 2.094969989053531e+152 2.094969989053531e+152 true
507 4.189939978107062e+152 4.189939978107062e+152 true
508 8.379879956214123e+152 8.379879956214123e+152 true
509 1.6759759912428246e+153 1.6759759912428246e+153 true
510 3.3519519824856493e+153 3.3519519824856493e+153 true
511 6.703903964971299e+153 6.703903964971299e+153 true
512 1.3407807929942597e+154 1.3407807929942597e+154 true
===*/

/* Exactness tests.  Floating point value parsing is required to be bit
 * exact.
 *
 * How to test for exact numbers?  Base 2 operations are easy to rely on
 * being correct (even parseInt()) so we can create comparison values in
 * many cases that way.
 */

print('exactness');

/*
# repr() apparently ensures enough decimals are printed to guarantee
# exact reconstruction of IEEE double
print('[');
for i in xrange(512+1):
  v1 = 2**i     # exact in python
  v2 = float(v1)
  print('{ exp: %d, dbl: "%r" },' % (i, v2))

print(']');
*/

function exactnessTest() {
    var two_to_n = [
        { exp: 0, dbl: "1.0" },
        { exp: 1, dbl: "2.0" },
        { exp: 2, dbl: "4.0" },
        { exp: 3, dbl: "8.0" },
        { exp: 4, dbl: "16.0" },
        { exp: 5, dbl: "32.0" },
        { exp: 6, dbl: "64.0" },
        { exp: 7, dbl: "128.0" },
        { exp: 8, dbl: "256.0" },
        { exp: 9, dbl: "512.0" },
        { exp: 10, dbl: "1024.0" },
        { exp: 11, dbl: "2048.0" },
        { exp: 12, dbl: "4096.0" },
        { exp: 13, dbl: "8192.0" },
        { exp: 14, dbl: "16384.0" },
        { exp: 15, dbl: "32768.0" },
        { exp: 16, dbl: "65536.0" },
        { exp: 17, dbl: "131072.0" },
        { exp: 18, dbl: "262144.0" },
        { exp: 19, dbl: "524288.0" },
        { exp: 20, dbl: "1048576.0" },
        { exp: 21, dbl: "2097152.0" },
        { exp: 22, dbl: "4194304.0" },
        { exp: 23, dbl: "8388608.0" },
        { exp: 24, dbl: "16777216.0" },
        { exp: 25, dbl: "33554432.0" },
        { exp: 26, dbl: "67108864.0" },
        { exp: 27, dbl: "134217728.0" },
        { exp: 28, dbl: "268435456.0" },
        { exp: 29, dbl: "536870912.0" },
        { exp: 30, dbl: "1073741824.0" },
        { exp: 31, dbl: "2147483648.0" },
        { exp: 32, dbl: "4294967296.0" },
        { exp: 33, dbl: "8589934592.0" },
        { exp: 34, dbl: "17179869184.0" },
        { exp: 35, dbl: "34359738368.0" },
        { exp: 36, dbl: "68719476736.0" },
        { exp: 37, dbl: "137438953472.0" },
        { exp: 38, dbl: "274877906944.0" },
        { exp: 39, dbl: "549755813888.0" },
        { exp: 40, dbl: "1099511627776.0" },
        { exp: 41, dbl: "2199023255552.0" },
        { exp: 42, dbl: "4398046511104.0" },
        { exp: 43, dbl: "8796093022208.0" },
        { exp: 44, dbl: "17592186044416.0" },
        { exp: 45, dbl: "35184372088832.0" },
        { exp: 46, dbl: "70368744177664.0" },
        { exp: 47, dbl: "140737488355328.0" },
        { exp: 48, dbl: "281474976710656.0" },
        { exp: 49, dbl: "562949953421312.0" },
        { exp: 50, dbl: "1125899906842624.0" },
        { exp: 51, dbl: "2251799813685248.0" },
        { exp: 52, dbl: "4503599627370496.0" },
        { exp: 53, dbl: "9007199254740992.0" },
        { exp: 54, dbl: "1.8014398509481984e+16" },
        { exp: 55, dbl: "3.602879701896397e+16" },
        { exp: 56, dbl: "7.205759403792794e+16" },
        { exp: 57, dbl: "1.4411518807585587e+17" },
        { exp: 58, dbl: "2.8823037615171174e+17" },
        { exp: 59, dbl: "5.764607523034235e+17" },
        { exp: 60, dbl: "1.152921504606847e+18" },
        { exp: 61, dbl: "2.305843009213694e+18" },
        { exp: 62, dbl: "4.611686018427388e+18" },
        { exp: 63, dbl: "9.223372036854776e+18" },
        { exp: 64, dbl: "1.8446744073709552e+19" },
        { exp: 65, dbl: "3.6893488147419103e+19" },
        { exp: 66, dbl: "7.378697629483821e+19" },
        { exp: 67, dbl: "1.4757395258967641e+20" },
        { exp: 68, dbl: "2.9514790517935283e+20" },
        { exp: 69, dbl: "5.902958103587057e+20" },
        { exp: 70, dbl: "1.1805916207174113e+21" },
        { exp: 71, dbl: "2.3611832414348226e+21" },
        { exp: 72, dbl: "4.722366482869645e+21" },
        { exp: 73, dbl: "9.44473296573929e+21" },
        { exp: 74, dbl: "1.888946593147858e+22" },
        { exp: 75, dbl: "3.777893186295716e+22" },
        { exp: 76, dbl: "7.555786372591432e+22" },
        { exp: 77, dbl: "1.5111572745182865e+23" },
        { exp: 78, dbl: "3.022314549036573e+23" },
        { exp: 79, dbl: "6.044629098073146e+23" },
        { exp: 80, dbl: "1.2089258196146292e+24" },
        { exp: 81, dbl: "2.4178516392292583e+24" },
        { exp: 82, dbl: "4.835703278458517e+24" },
        { exp: 83, dbl: "9.671406556917033e+24" },
        { exp: 84, dbl: "1.9342813113834067e+25" },
        { exp: 85, dbl: "3.8685626227668134e+25" },
        { exp: 86, dbl: "7.737125245533627e+25" },
        { exp: 87, dbl: "1.5474250491067253e+26" },
        { exp: 88, dbl: "3.094850098213451e+26" },
        { exp: 89, dbl: "6.189700196426902e+26" },
        { exp: 90, dbl: "1.2379400392853803e+27" },
        { exp: 91, dbl: "2.4758800785707605e+27" },
        { exp: 92, dbl: "4.951760157141521e+27" },
        { exp: 93, dbl: "9.903520314283042e+27" },
        { exp: 94, dbl: "1.9807040628566084e+28" },
        { exp: 95, dbl: "3.961408125713217e+28" },
        { exp: 96, dbl: "7.922816251426434e+28" },
        { exp: 97, dbl: "1.5845632502852868e+29" },
        { exp: 98, dbl: "3.1691265005705735e+29" },
        { exp: 99, dbl: "6.338253001141147e+29" },
        { exp: 100, dbl: "1.2676506002282294e+30" },
        { exp: 101, dbl: "2.535301200456459e+30" },
        { exp: 102, dbl: "5.070602400912918e+30" },
        { exp: 103, dbl: "1.0141204801825835e+31" },
        { exp: 104, dbl: "2.028240960365167e+31" },
        { exp: 105, dbl: "4.056481920730334e+31" },
        { exp: 106, dbl: "8.112963841460668e+31" },
        { exp: 107, dbl: "1.6225927682921336e+32" },
        { exp: 108, dbl: "3.2451855365842673e+32" },
        { exp: 109, dbl: "6.490371073168535e+32" },
        { exp: 110, dbl: "1.298074214633707e+33" },
        { exp: 111, dbl: "2.596148429267414e+33" },
        { exp: 112, dbl: "5.192296858534828e+33" },
        { exp: 113, dbl: "1.0384593717069655e+34" },
        { exp: 114, dbl: "2.076918743413931e+34" },
        { exp: 115, dbl: "4.153837486827862e+34" },
        { exp: 116, dbl: "8.307674973655724e+34" },
        { exp: 117, dbl: "1.661534994731145e+35" },
        { exp: 118, dbl: "3.32306998946229e+35" },
        { exp: 119, dbl: "6.64613997892458e+35" },
        { exp: 120, dbl: "1.329227995784916e+36" },
        { exp: 121, dbl: "2.658455991569832e+36" },
        { exp: 122, dbl: "5.316911983139664e+36" },
        { exp: 123, dbl: "1.0633823966279327e+37" },
        { exp: 124, dbl: "2.1267647932558654e+37" },
        { exp: 125, dbl: "4.253529586511731e+37" },
        { exp: 126, dbl: "8.507059173023462e+37" },
        { exp: 127, dbl: "1.7014118346046923e+38" },
        { exp: 128, dbl: "3.402823669209385e+38" },
        { exp: 129, dbl: "6.80564733841877e+38" },
        { exp: 130, dbl: "1.361129467683754e+39" },
        { exp: 131, dbl: "2.722258935367508e+39" },
        { exp: 132, dbl: "5.444517870735016e+39" },
        { exp: 133, dbl: "1.0889035741470031e+40" },
        { exp: 134, dbl: "2.1778071482940062e+40" },
        { exp: 135, dbl: "4.3556142965880123e+40" },
        { exp: 136, dbl: "8.711228593176025e+40" },
        { exp: 137, dbl: "1.742245718635205e+41" },
        { exp: 138, dbl: "3.48449143727041e+41" },
        { exp: 139, dbl: "6.96898287454082e+41" },
        { exp: 140, dbl: "1.393796574908164e+42" },
        { exp: 141, dbl: "2.787593149816328e+42" },
        { exp: 142, dbl: "5.575186299632656e+42" },
        { exp: 143, dbl: "1.1150372599265312e+43" },
        { exp: 144, dbl: "2.2300745198530623e+43" },
        { exp: 145, dbl: "4.460149039706125e+43" },
        { exp: 146, dbl: "8.92029807941225e+43" },
        { exp: 147, dbl: "1.78405961588245e+44" },
        { exp: 148, dbl: "3.5681192317649e+44" },
        { exp: 149, dbl: "7.1362384635298e+44" },
        { exp: 150, dbl: "1.42724769270596e+45" },
        { exp: 151, dbl: "2.85449538541192e+45" },
        { exp: 152, dbl: "5.70899077082384e+45" },
        { exp: 153, dbl: "1.141798154164768e+46" },
        { exp: 154, dbl: "2.283596308329536e+46" },
        { exp: 155, dbl: "4.567192616659072e+46" },
        { exp: 156, dbl: "9.134385233318143e+46" },
        { exp: 157, dbl: "1.8268770466636286e+47" },
        { exp: 158, dbl: "3.6537540933272573e+47" },
        { exp: 159, dbl: "7.307508186654515e+47" },
        { exp: 160, dbl: "1.461501637330903e+48" },
        { exp: 161, dbl: "2.923003274661806e+48" },
        { exp: 162, dbl: "5.846006549323612e+48" },
        { exp: 163, dbl: "1.1692013098647223e+49" },
        { exp: 164, dbl: "2.3384026197294447e+49" },
        { exp: 165, dbl: "4.6768052394588893e+49" },
        { exp: 166, dbl: "9.353610478917779e+49" },
        { exp: 167, dbl: "1.8707220957835557e+50" },
        { exp: 168, dbl: "3.7414441915671115e+50" },
        { exp: 169, dbl: "7.482888383134223e+50" },
        { exp: 170, dbl: "1.4965776766268446e+51" },
        { exp: 171, dbl: "2.9931553532536892e+51" },
        { exp: 172, dbl: "5.986310706507379e+51" },
        { exp: 173, dbl: "1.1972621413014757e+52" },
        { exp: 174, dbl: "2.3945242826029513e+52" },
        { exp: 175, dbl: "4.789048565205903e+52" },
        { exp: 176, dbl: "9.578097130411805e+52" },
        { exp: 177, dbl: "1.915619426082361e+53" },
        { exp: 178, dbl: "3.831238852164722e+53" },
        { exp: 179, dbl: "7.662477704329444e+53" },
        { exp: 180, dbl: "1.532495540865889e+54" },
        { exp: 181, dbl: "3.064991081731778e+54" },
        { exp: 182, dbl: "6.129982163463556e+54" },
        { exp: 183, dbl: "1.2259964326927111e+55" },
        { exp: 184, dbl: "2.4519928653854222e+55" },
        { exp: 185, dbl: "4.9039857307708443e+55" },
        { exp: 186, dbl: "9.807971461541689e+55" },
        { exp: 187, dbl: "1.9615942923083377e+56" },
        { exp: 188, dbl: "3.9231885846166755e+56" },
        { exp: 189, dbl: "7.846377169233351e+56" },
        { exp: 190, dbl: "1.5692754338466702e+57" },
        { exp: 191, dbl: "3.1385508676933404e+57" },
        { exp: 192, dbl: "6.277101735386681e+57" },
        { exp: 193, dbl: "1.2554203470773362e+58" },
        { exp: 194, dbl: "2.5108406941546723e+58" },
        { exp: 195, dbl: "5.021681388309345e+58" },
        { exp: 196, dbl: "1.004336277661869e+59" },
        { exp: 197, dbl: "2.008672555323738e+59" },
        { exp: 198, dbl: "4.017345110647476e+59" },
        { exp: 199, dbl: "8.034690221294951e+59" },
        { exp: 200, dbl: "1.6069380442589903e+60" },
        { exp: 201, dbl: "3.2138760885179806e+60" },
        { exp: 202, dbl: "6.427752177035961e+60" },
        { exp: 203, dbl: "1.2855504354071922e+61" },
        { exp: 204, dbl: "2.5711008708143844e+61" },
        { exp: 205, dbl: "5.142201741628769e+61" },
        { exp: 206, dbl: "1.0284403483257538e+62" },
        { exp: 207, dbl: "2.0568806966515076e+62" },
        { exp: 208, dbl: "4.113761393303015e+62" },
        { exp: 209, dbl: "8.22752278660603e+62" },
        { exp: 210, dbl: "1.645504557321206e+63" },
        { exp: 211, dbl: "3.291009114642412e+63" },
        { exp: 212, dbl: "6.582018229284824e+63" },
        { exp: 213, dbl: "1.3164036458569648e+64" },
        { exp: 214, dbl: "2.6328072917139297e+64" },
        { exp: 215, dbl: "5.2656145834278593e+64" },
        { exp: 216, dbl: "1.0531229166855719e+65" },
        { exp: 217, dbl: "2.1062458333711437e+65" },
        { exp: 218, dbl: "4.2124916667422875e+65" },
        { exp: 219, dbl: "8.424983333484575e+65" },
        { exp: 220, dbl: "1.684996666696915e+66" },
        { exp: 221, dbl: "3.36999333339383e+66" },
        { exp: 222, dbl: "6.73998666678766e+66" },
        { exp: 223, dbl: "1.347997333357532e+67" },
        { exp: 224, dbl: "2.695994666715064e+67" },
        { exp: 225, dbl: "5.391989333430128e+67" },
        { exp: 226, dbl: "1.0783978666860256e+68" },
        { exp: 227, dbl: "2.1567957333720512e+68" },
        { exp: 228, dbl: "4.3135914667441024e+68" },
        { exp: 229, dbl: "8.627182933488205e+68" },
        { exp: 230, dbl: "1.725436586697641e+69" },
        { exp: 231, dbl: "3.450873173395282e+69" },
        { exp: 232, dbl: "6.901746346790564e+69" },
        { exp: 233, dbl: "1.3803492693581128e+70" },
        { exp: 234, dbl: "2.7606985387162255e+70" },
        { exp: 235, dbl: "5.521397077432451e+70" },
        { exp: 236, dbl: "1.1042794154864902e+71" },
        { exp: 237, dbl: "2.2085588309729804e+71" },
        { exp: 238, dbl: "4.417117661945961e+71" },
        { exp: 239, dbl: "8.834235323891922e+71" },
        { exp: 240, dbl: "1.7668470647783843e+72" },
        { exp: 241, dbl: "3.533694129556769e+72" },
        { exp: 242, dbl: "7.067388259113537e+72" },
        { exp: 243, dbl: "1.4134776518227075e+73" },
        { exp: 244, dbl: "2.826955303645415e+73" },
        { exp: 245, dbl: "5.65391060729083e+73" },
        { exp: 246, dbl: "1.130782121458166e+74" },
        { exp: 247, dbl: "2.261564242916332e+74" },
        { exp: 248, dbl: "4.523128485832664e+74" },
        { exp: 249, dbl: "9.046256971665328e+74" },
        { exp: 250, dbl: "1.8092513943330656e+75" },
        { exp: 251, dbl: "3.618502788666131e+75" },
        { exp: 252, dbl: "7.237005577332262e+75" },
        { exp: 253, dbl: "1.4474011154664524e+76" },
        { exp: 254, dbl: "2.894802230932905e+76" },
        { exp: 255, dbl: "5.78960446186581e+76" },
        { exp: 256, dbl: "1.157920892373162e+77" },
        { exp: 257, dbl: "2.315841784746324e+77" },
        { exp: 258, dbl: "4.631683569492648e+77" },
        { exp: 259, dbl: "9.263367138985296e+77" },
        { exp: 260, dbl: "1.8526734277970591e+78" },
        { exp: 261, dbl: "3.7053468555941183e+78" },
        { exp: 262, dbl: "7.410693711188237e+78" },
        { exp: 263, dbl: "1.4821387422376473e+79" },
        { exp: 264, dbl: "2.9642774844752946e+79" },
        { exp: 265, dbl: "5.928554968950589e+79" },
        { exp: 266, dbl: "1.1857109937901178e+80" },
        { exp: 267, dbl: "2.3714219875802357e+80" },
        { exp: 268, dbl: "4.7428439751604714e+80" },
        { exp: 269, dbl: "9.485687950320943e+80" },
        { exp: 270, dbl: "1.8971375900641885e+81" },
        { exp: 271, dbl: "3.794275180128377e+81" },
        { exp: 272, dbl: "7.588550360256754e+81" },
        { exp: 273, dbl: "1.517710072051351e+82" },
        { exp: 274, dbl: "3.035420144102702e+82" },
        { exp: 275, dbl: "6.070840288205404e+82" },
        { exp: 276, dbl: "1.2141680576410807e+83" },
        { exp: 277, dbl: "2.4283361152821613e+83" },
        { exp: 278, dbl: "4.856672230564323e+83" },
        { exp: 279, dbl: "9.713344461128645e+83" },
        { exp: 280, dbl: "1.942668892225729e+84" },
        { exp: 281, dbl: "3.885337784451458e+84" },
        { exp: 282, dbl: "7.770675568902916e+84" },
        { exp: 283, dbl: "1.5541351137805833e+85" },
        { exp: 284, dbl: "3.1082702275611665e+85" },
        { exp: 285, dbl: "6.216540455122333e+85" },
        { exp: 286, dbl: "1.2433080910244666e+86" },
        { exp: 287, dbl: "2.4866161820489332e+86" },
        { exp: 288, dbl: "4.9732323640978664e+86" },
        { exp: 289, dbl: "9.946464728195733e+86" },
        { exp: 290, dbl: "1.9892929456391466e+87" },
        { exp: 291, dbl: "3.978585891278293e+87" },
        { exp: 292, dbl: "7.957171782556586e+87" },
        { exp: 293, dbl: "1.5914343565113173e+88" },
        { exp: 294, dbl: "3.1828687130226345e+88" },
        { exp: 295, dbl: "6.365737426045269e+88" },
        { exp: 296, dbl: "1.2731474852090538e+89" },
        { exp: 297, dbl: "2.5462949704181076e+89" },
        { exp: 298, dbl: "5.092589940836215e+89" },
        { exp: 299, dbl: "1.018517988167243e+90" },
        { exp: 300, dbl: "2.037035976334486e+90" },
        { exp: 301, dbl: "4.074071952668972e+90" },
        { exp: 302, dbl: "8.148143905337944e+90" },
        { exp: 303, dbl: "1.629628781067589e+91" },
        { exp: 304, dbl: "3.259257562135178e+91" },
        { exp: 305, dbl: "6.518515124270356e+91" },
        { exp: 306, dbl: "1.3037030248540711e+92" },
        { exp: 307, dbl: "2.6074060497081422e+92" },
        { exp: 308, dbl: "5.2148120994162844e+92" },
        { exp: 309, dbl: "1.0429624198832569e+93" },
        { exp: 310, dbl: "2.0859248397665138e+93" },
        { exp: 311, dbl: "4.1718496795330275e+93" },
        { exp: 312, dbl: "8.343699359066055e+93" },
        { exp: 313, dbl: "1.668739871813211e+94" },
        { exp: 314, dbl: "3.337479743626422e+94" },
        { exp: 315, dbl: "6.674959487252844e+94" },
        { exp: 316, dbl: "1.3349918974505688e+95" },
        { exp: 317, dbl: "2.6699837949011376e+95" },
        { exp: 318, dbl: "5.339967589802275e+95" },
        { exp: 319, dbl: "1.067993517960455e+96" },
        { exp: 320, dbl: "2.13598703592091e+96" },
        { exp: 321, dbl: "4.27197407184182e+96" },
        { exp: 322, dbl: "8.54394814368364e+96" },
        { exp: 323, dbl: "1.708789628736728e+97" },
        { exp: 324, dbl: "3.417579257473456e+97" },
        { exp: 325, dbl: "6.835158514946912e+97" },
        { exp: 326, dbl: "1.3670317029893825e+98" },
        { exp: 327, dbl: "2.734063405978765e+98" },
        { exp: 328, dbl: "5.46812681195753e+98" },
        { exp: 329, dbl: "1.093625362391506e+99" },
        { exp: 330, dbl: "2.187250724783012e+99" },
        { exp: 331, dbl: "4.374501449566024e+99" },
        { exp: 332, dbl: "8.749002899132048e+99" },
        { exp: 333, dbl: "1.7498005798264095e+100" },
        { exp: 334, dbl: "3.499601159652819e+100" },
        { exp: 335, dbl: "6.999202319305638e+100" },
        { exp: 336, dbl: "1.3998404638611276e+101" },
        { exp: 337, dbl: "2.7996809277222553e+101" },
        { exp: 338, dbl: "5.599361855444511e+101" },
        { exp: 339, dbl: "1.1198723710889021e+102" },
        { exp: 340, dbl: "2.2397447421778042e+102" },
        { exp: 341, dbl: "4.4794894843556084e+102" },
        { exp: 342, dbl: "8.958978968711217e+102" },
        { exp: 343, dbl: "1.7917957937422434e+103" },
        { exp: 344, dbl: "3.583591587484487e+103" },
        { exp: 345, dbl: "7.167183174968974e+103" },
        { exp: 346, dbl: "1.4334366349937947e+104" },
        { exp: 347, dbl: "2.8668732699875894e+104" },
        { exp: 348, dbl: "5.733746539975179e+104" },
        { exp: 349, dbl: "1.1467493079950358e+105" },
        { exp: 350, dbl: "2.2934986159900715e+105" },
        { exp: 351, dbl: "4.586997231980143e+105" },
        { exp: 352, dbl: "9.173994463960286e+105" },
        { exp: 353, dbl: "1.8347988927920572e+106" },
        { exp: 354, dbl: "3.6695977855841144e+106" },
        { exp: 355, dbl: "7.339195571168229e+106" },
        { exp: 356, dbl: "1.4678391142336458e+107" },
        { exp: 357, dbl: "2.9356782284672915e+107" },
        { exp: 358, dbl: "5.871356456934583e+107" },
        { exp: 359, dbl: "1.1742712913869166e+108" },
        { exp: 360, dbl: "2.3485425827738332e+108" },
        { exp: 361, dbl: "4.6970851655476665e+108" },
        { exp: 362, dbl: "9.394170331095333e+108" },
        { exp: 363, dbl: "1.8788340662190666e+109" },
        { exp: 364, dbl: "3.757668132438133e+109" },
        { exp: 365, dbl: "7.515336264876266e+109" },
        { exp: 366, dbl: "1.5030672529752533e+110" },
        { exp: 367, dbl: "3.0061345059505065e+110" },
        { exp: 368, dbl: "6.012269011901013e+110" },
        { exp: 369, dbl: "1.2024538023802026e+111" },
        { exp: 370, dbl: "2.4049076047604052e+111" },
        { exp: 371, dbl: "4.8098152095208105e+111" },
        { exp: 372, dbl: "9.619630419041621e+111" },
        { exp: 373, dbl: "1.9239260838083242e+112" },
        { exp: 374, dbl: "3.8478521676166484e+112" },
        { exp: 375, dbl: "7.695704335233297e+112" },
        { exp: 376, dbl: "1.5391408670466593e+113" },
        { exp: 377, dbl: "3.078281734093319e+113" },
        { exp: 378, dbl: "6.156563468186638e+113" },
        { exp: 379, dbl: "1.2313126936373275e+114" },
        { exp: 380, dbl: "2.462625387274655e+114" },
        { exp: 381, dbl: "4.92525077454931e+114" },
        { exp: 382, dbl: "9.85050154909862e+114" },
        { exp: 383, dbl: "1.970100309819724e+115" },
        { exp: 384, dbl: "3.940200619639448e+115" },
        { exp: 385, dbl: "7.880401239278896e+115" },
        { exp: 386, dbl: "1.5760802478557792e+116" },
        { exp: 387, dbl: "3.1521604957115583e+116" },
        { exp: 388, dbl: "6.304320991423117e+116" },
        { exp: 389, dbl: "1.2608641982846233e+117" },
        { exp: 390, dbl: "2.5217283965692467e+117" },
        { exp: 391, dbl: "5.0434567931384933e+117" },
        { exp: 392, dbl: "1.0086913586276987e+118" },
        { exp: 393, dbl: "2.0173827172553973e+118" },
        { exp: 394, dbl: "4.034765434510795e+118" },
        { exp: 395, dbl: "8.06953086902159e+118" },
        { exp: 396, dbl: "1.613906173804318e+119" },
        { exp: 397, dbl: "3.227812347608636e+119" },
        { exp: 398, dbl: "6.455624695217272e+119" },
        { exp: 399, dbl: "1.2911249390434543e+120" },
        { exp: 400, dbl: "2.5822498780869086e+120" },
        { exp: 401, dbl: "5.164499756173817e+120" },
        { exp: 402, dbl: "1.0328999512347634e+121" },
        { exp: 403, dbl: "2.065799902469527e+121" },
        { exp: 404, dbl: "4.131599804939054e+121" },
        { exp: 405, dbl: "8.263199609878108e+121" },
        { exp: 406, dbl: "1.6526399219756215e+122" },
        { exp: 407, dbl: "3.305279843951243e+122" },
        { exp: 408, dbl: "6.610559687902486e+122" },
        { exp: 409, dbl: "1.3221119375804972e+123" },
        { exp: 410, dbl: "2.6442238751609944e+123" },
        { exp: 411, dbl: "5.288447750321989e+123" },
        { exp: 412, dbl: "1.0576895500643978e+124" },
        { exp: 413, dbl: "2.1153791001287955e+124" },
        { exp: 414, dbl: "4.230758200257591e+124" },
        { exp: 415, dbl: "8.461516400515182e+124" },
        { exp: 416, dbl: "1.6923032801030364e+125" },
        { exp: 417, dbl: "3.384606560206073e+125" },
        { exp: 418, dbl: "6.769213120412146e+125" },
        { exp: 419, dbl: "1.3538426240824291e+126" },
        { exp: 420, dbl: "2.7076852481648583e+126" },
        { exp: 421, dbl: "5.415370496329717e+126" },
        { exp: 422, dbl: "1.0830740992659433e+127" },
        { exp: 423, dbl: "2.1661481985318866e+127" },
        { exp: 424, dbl: "4.332296397063773e+127" },
        { exp: 425, dbl: "8.664592794127546e+127" },
        { exp: 426, dbl: "1.7329185588255093e+128" },
        { exp: 427, dbl: "3.4658371176510186e+128" },
        { exp: 428, dbl: "6.931674235302037e+128" },
        { exp: 429, dbl: "1.3863348470604074e+129" },
        { exp: 430, dbl: "2.772669694120815e+129" },
        { exp: 431, dbl: "5.54533938824163e+129" },
        { exp: 432, dbl: "1.109067877648326e+130" },
        { exp: 433, dbl: "2.218135755296652e+130" },
        { exp: 434, dbl: "4.436271510593304e+130" },
        { exp: 435, dbl: "8.872543021186608e+130" },
        { exp: 436, dbl: "1.7745086042373215e+131" },
        { exp: 437, dbl: "3.549017208474643e+131" },
        { exp: 438, dbl: "7.098034416949286e+131" },
        { exp: 439, dbl: "1.4196068833898572e+132" },
        { exp: 440, dbl: "2.8392137667797144e+132" },
        { exp: 441, dbl: "5.678427533559429e+132" },
        { exp: 442, dbl: "1.1356855067118858e+133" },
        { exp: 443, dbl: "2.2713710134237715e+133" },
        { exp: 444, dbl: "4.542742026847543e+133" },
        { exp: 445, dbl: "9.085484053695086e+133" },
        { exp: 446, dbl: "1.8170968107390172e+134" },
        { exp: 447, dbl: "3.6341936214780345e+134" },
        { exp: 448, dbl: "7.268387242956069e+134" },
        { exp: 449, dbl: "1.4536774485912138e+135" },
        { exp: 450, dbl: "2.9073548971824276e+135" },
        { exp: 451, dbl: "5.814709794364855e+135" },
        { exp: 452, dbl: "1.162941958872971e+136" },
        { exp: 453, dbl: "2.325883917745942e+136" },
        { exp: 454, dbl: "4.651767835491884e+136" },
        { exp: 455, dbl: "9.303535670983768e+136" },
        { exp: 456, dbl: "1.8607071341967536e+137" },
        { exp: 457, dbl: "3.7214142683935073e+137" },
        { exp: 458, dbl: "7.442828536787015e+137" },
        { exp: 459, dbl: "1.488565707357403e+138" },
        { exp: 460, dbl: "2.977131414714806e+138" },
        { exp: 461, dbl: "5.954262829429612e+138" },
        { exp: 462, dbl: "1.1908525658859223e+139" },
        { exp: 463, dbl: "2.3817051317718447e+139" },
        { exp: 464, dbl: "4.7634102635436893e+139" },
        { exp: 465, dbl: "9.526820527087379e+139" },
        { exp: 466, dbl: "1.9053641054174757e+140" },
        { exp: 467, dbl: "3.8107282108349515e+140" },
        { exp: 468, dbl: "7.621456421669903e+140" },
        { exp: 469, dbl: "1.5242912843339806e+141" },
        { exp: 470, dbl: "3.048582568667961e+141" },
        { exp: 471, dbl: "6.097165137335922e+141" },
        { exp: 472, dbl: "1.2194330274671845e+142" },
        { exp: 473, dbl: "2.438866054934369e+142" },
        { exp: 474, dbl: "4.877732109868738e+142" },
        { exp: 475, dbl: "9.755464219737476e+142" },
        { exp: 476, dbl: "1.9510928439474951e+143" },
        { exp: 477, dbl: "3.9021856878949903e+143" },
        { exp: 478, dbl: "7.804371375789981e+143" },
        { exp: 479, dbl: "1.5608742751579961e+144" },
        { exp: 480, dbl: "3.1217485503159922e+144" },
        { exp: 481, dbl: "6.243497100631985e+144" },
        { exp: 482, dbl: "1.248699420126397e+145" },
        { exp: 483, dbl: "2.497398840252794e+145" },
        { exp: 484, dbl: "4.994797680505588e+145" },
        { exp: 485, dbl: "9.989595361011175e+145" },
        { exp: 486, dbl: "1.997919072202235e+146" },
        { exp: 487, dbl: "3.99583814440447e+146" },
        { exp: 488, dbl: "7.99167628880894e+146" },
        { exp: 489, dbl: "1.598335257761788e+147" },
        { exp: 490, dbl: "3.196670515523576e+147" },
        { exp: 491, dbl: "6.393341031047152e+147" },
        { exp: 492, dbl: "1.2786682062094304e+148" },
        { exp: 493, dbl: "2.557336412418861e+148" },
        { exp: 494, dbl: "5.114672824837722e+148" },
        { exp: 495, dbl: "1.0229345649675443e+149" },
        { exp: 496, dbl: "2.0458691299350887e+149" },
        { exp: 497, dbl: "4.0917382598701773e+149" },
        { exp: 498, dbl: "8.183476519740355e+149" },
        { exp: 499, dbl: "1.636695303948071e+150" },
        { exp: 500, dbl: "3.273390607896142e+150" },
        { exp: 501, dbl: "6.546781215792284e+150" },
        { exp: 502, dbl: "1.3093562431584567e+151" },
        { exp: 503, dbl: "2.6187124863169135e+151" },
        { exp: 504, dbl: "5.237424972633827e+151" },
        { exp: 505, dbl: "1.0474849945267654e+152" },
        { exp: 506, dbl: "2.094969989053531e+152" },
        { exp: 507, dbl: "4.189939978107062e+152" },
        { exp: 508, dbl: "8.379879956214123e+152" },
        { exp: 509, dbl: "1.6759759912428246e+153" },
        { exp: 510, dbl: "3.3519519824856493e+153" },
        { exp: 511, dbl: "6.703903964971299e+153" },
        { exp: 512, dbl: "1.3407807929942597e+154" },
    ];
    var i, j;
    var t;
    var o;
    var d1;
    var d2;

    for (i = 0; i < two_to_n.length; i++) {
        o = two_to_n[i];
        t = '1';
        for (j = 0; j < o.exp; j++) {
            t = t + '0';
        }
        d1 = g.parseInt(t, 2);
        d2 = g.parseFloat(o.dbl);
        print(i, d1, d2, d1 === d2);
    }
}

try {
    exactnessTest();
} catch (e) {
    print(e.name);
}

/* XXX: more tests */
