/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_dom_PresentationRequest_h
#define mozilla_dom_PresentationRequest_h

#include "mozilla/dom/BindingDeclarations.h"
#include "mozilla/DOMEventTargetHelper.h"

namespace mozilla {
namespace dom {

class Promise;
class PresentationAvailability;
class PresentationConnection;

class PresentationRequest final : public DOMEventTargetHelper {
 public:
  NS_DECL_ISUPPORTS_INHERITED

  static already_AddRefed<PresentationRequest> Constructor(
      const GlobalObject& aGlobal, const nsAString& aUrl, ErrorResult& aRv);

  static already_AddRefed<PresentationRequest> Constructor(
      const GlobalObject& aGlobal, const Sequence<nsString>& aUrls,
      ErrorResult& aRv);

  virtual JSObject* WrapObject(JSContext* aCx,
                               JS::Handle<JSObject*> aGivenProto) override;

  // WebIDL (public APIs)
  already_AddRefed<Promise> Start(ErrorResult& aRv);

  already_AddRefed<Promise> StartWithDevice(const nsAString& aDeviceId,
                                            ErrorResult& aRv);

  already_AddRefed<Promise> Reconnect(const nsAString& aPresentationId,
                                      ErrorResult& aRv);

  already_AddRefed<Promise> GetAvailability(ErrorResult& aRv);

  IMPL_EVENT_HANDLER(connectionavailable);

  nsresult DispatchConnectionAvailableEvent(
      PresentationConnection* aConnection);

  void NotifyPromiseSettled();

 private:
  PresentationRequest(nsPIDOMWindowInner* aWindow, nsTArray<nsString>&& aUrls);

  ~PresentationRequest();

  bool Init();

  void FindOrCreatePresentationConnection(const nsAString& aPresentationId,
                                          Promise* aPromise);

  void FindOrCreatePresentationAvailability(RefPtr<Promise>& aPromise);

  // Implement
  // https://w3c.github.io/webappsec-mixed-content/#categorize-settings-object
  bool IsProhibitMixedSecurityContexts(Document* aDocument);

  // Implement
  // https://w3c.github.io/webappsec-mixed-content/#a-priori-authenticated-url
  bool IsPrioriAuthenticatedURL(const nsAString& aUrl);

  bool IsAllURLAuthenticated();

  nsTArray<nsString> mUrls;
};

}  // namespace dom
}  // namespace mozilla

#endif  // mozilla_dom_PresentationRequest_h
