// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * USB specification data
 *
 * Copyright (C) 2013 Tom Gundersen <teg@jklm.no>
 */

#ifndef _USB_SPEC_H
#define _USB_SPEC_H

/* ---------------------------------------------------------------------- */

struct audioterminal {
	const uint16_t termt;
	const char *name;
};

struct videoterminal {
	const uint16_t termt;
	const char *name;
};

struct genericstrtable {
	const unsigned int num;
	const char *name;
};

/* ---------------------------------------------------------------------- */

static const struct audioterminal audioterminals[] = {
	{ 0x0100, "USB Undefined" },
	{ 0x0101, "USB Streaming" },
	{ 0x01ff, "USB Vendor Specific" },
	{ 0x0200, "Input Undefined" },
	{ 0x0201, "Microphone" },
	{ 0x0202, "Desktop Microphone" },
	{ 0x0203, "Personal Microphone" },
	{ 0x0204, "Omni-directional Microphone" },
	{ 0x0205, "Microphone Array" },
	{ 0x0206, "Processing Microphone Array" },
	{ 0x0300, "Output Undefined" },
	{ 0x0301, "Speaker" },
	{ 0x0302, "Headphones" },
	{ 0x0303, "Head Mounted Display Audio" },
	{ 0x0304, "Desktop Speaker" },
	{ 0x0305, "Room Speaker" },
	{ 0x0306, "Communication Speaker" },
	{ 0x0307, "Low Frequency Effects Speaker" },
	{ 0x0400, "Bidirectional Undefined" },
	{ 0x0401, "Handset" },
	{ 0x0402, "Headset" },
	{ 0x0403, "Speakerphone, no echo reduction" },
	{ 0x0404, "Echo-suppressing speakerphone" },
	{ 0x0405, "Echo-canceling speakerphone" },
	{ 0x0500, "Telephony Undefined" },
	{ 0x0501, "Phone line" },
	{ 0x0502, "Telephone" },
	{ 0x0503, "Down Line Phone" },
	{ 0x0600, "External Undefined" },
	{ 0x0601, "Analog Connector" },
	{ 0x0602, "Digital Audio Interface" },
	{ 0x0603, "Line Connector" },
	{ 0x0604, "Legacy Audio Connector" },
	{ 0x0605, "SPDIF interface" },
	{ 0x0606, "1394 DA stream" },
	{ 0x0607, "1394 DV stream soundtrack" },
	{ 0x0700, "Embedded Undefined" },
	{ 0x0701, "Level Calibration Noise Source" },
	{ 0x0702, "Equalization Noise" },
	{ 0x0703, "CD Player" },
	{ 0x0704, "DAT" },
	{ 0x0705, "DCC" },
	{ 0x0706, "MiniDisc" },
	{ 0x0707, "Analog Tape" },
	{ 0x0708, "Phonograph" },
	{ 0x0709, "VCR Audio" },
	{ 0x070a, "Video Disc Audio" },
	{ 0x070b, "DVD Audio" },
	{ 0x070c, "TV Tuner Audio" },
	{ 0x070d, "Satellite Receiver Audio" },
	{ 0x070e, "Cable Tuner Audio" },
	{ 0x070f, "DSS Audio" },
	{ 0x0710, "Radio Receiver" },
	{ 0x0711, "Radio Transmitter" },
	{ 0x0712, "Multitrack Recorder" },
	{ 0x0713, "Synthesizer" },
	{ 0x0000, NULL },
};

static const struct videoterminal videoterminals[] = {
	{ 0x0100, "USB Vendor Specific" },
	{ 0x0101, "USB Streaming" },
	{ 0x0200, "Input Vendor Specific" },
	{ 0x0201, "Camera Sensor" },
	{ 0x0202, "Sequential Media" },
	{ 0x0300, "Output Vendor Specific" },
	{ 0x0301, "Generic Display" },
	{ 0x0302, "Sequential Media" },
	{ 0x0400, "External Vendor Specific" },
	{ 0x0401, "Composite Video" },
	{ 0x0402, "S-Video" },
	{ 0x0403, "Component Video" },
	{ 0x0000, NULL },
};

static const struct genericstrtable hiddescriptors[] = {
	{ 0x21, "HID" },
	{ 0x22, "Report" },
	{ 0x23, "Physical" },
	{ 0x00, NULL },
};

static const struct genericstrtable reports[] = {
	{ 0x04, "Usage Page" },
	{ 0x08, "Usage" },
	{ 0x14, "Logical Minimum" },
	{ 0x18, "Usage Minimum" },
	{ 0x24, "Logical Maximum" },
	{ 0x28, "Usage Maximum" },
	{ 0x34, "Physical Minimum" },
	{ 0x38, "Designator Index" },
	{ 0x44, "Physical Maximum" },
	{ 0x48, "Designator Minimum" },
	{ 0x54, "Unit Exponent" },
	{ 0x58, "Designator Maximum" },
	{ 0x64, "Unit" },
	{ 0x74, "Report Size" },
	{ 0x78, "String Index" },
	{ 0x80, "Input" },
	{ 0x84, "Report ID" },
	{ 0x88, "String Minimum" },
	{ 0x90, "Output" },
	{ 0x94, "Report Count" },
	{ 0x98, "String Maximum" },
	{ 0xa0, "Collection" },
	{ 0xa4, "Push" },
	{ 0xa8, "Delimiter" },
	{ 0xb0, "Feature" },
	{ 0xb4, "Pop" },
	{ 0xc0, "End Collection" },
	{ 0x00, NULL },
};

static const struct genericstrtable huts[] = {
	{ 0x00, "Undefined" },
	{ 0x01, "Generic Desktop Controls" },
	{ 0x02, "Simulation Controls" },
	{ 0x03, "VR Controls" },
	{ 0x04, "Sport Controls" },
	{ 0x05, "Game Controls" },
	{ 0x07, "Keyboard" },
	{ 0x08, "LEDs" },
	{ 0x09, "Buttons" },
	{ 0x0a, "Ordinal" },
	{ 0x0b, "Telephony" },
	{ 0x0c, "Consumer" },
	{ 0x0d, "Digitizer" },
	{ 0x0f, "PID Page" },
	{ 0x10, "Unicode" },
	{ 0x14, "Alphanumeric Display" },
	{ 0x80, "USB Monitor" },
	{ 0x81, "USB Monitor Enumerated Values" },
	{ 0x82, "Monitor VESA Virtual Controls" },
	{ 0x84, "Power Device Page" },
	{ 0x85, "Battery System Page" },
	{ 0x86, "Power Pages" },
	{ 0x87, "Power Pages" },
	{ 0x8c, "Bar Code Scanner Page (POS)" },
	{ 0x8d, "Scale Page (POS)" },
	{ 0x90, "Camera Control Page" },
	{ 0x91, "Arcade Control Page" },
	{ 0xf0, "Cash Device" },
	{ 0x00, NULL },
};

static const struct genericstrtable biass[] = {
	// clang-format off
	{ 0x0, "Not Applicable" },
	{ 0x1, "Right Hand" },
	{ 0x2, "Left Hand" },
	{ 0x3, "Both Hands" },
	{ 0x4, "Either Hand" },
	{ 0x0, NULL },
	// clang-format on
};

static const struct genericstrtable physdess[] = {
	// clang-format off
	{ 0x00, "None" },
	{ 0x01, "Hand" },
	{ 0x02, "Eyeball" },
	{ 0x03, "Eyebrow" },
	{ 0x04, "Eyelid" },
	{ 0x05, "Ear" },
	{ 0x06, "Nose" },
	{ 0x07, "Mouth" },
	{ 0x08, "Upper Lip" },
	{ 0x09, "Lower Lip" },
	{ 0x0a, "Jaw" },
	{ 0x0b, "Neck" },
	{ 0x0c, "Upper Arm" },
	{ 0x0d, "Elbow" },
	{ 0x0e, "Forearm" },
	{ 0x0f, "Wrist" },
	{ 0x10, "Palm" },
	{ 0x11, "Thumb" },
	{ 0x12, "Index Finger" },
	{ 0x13, "Middle Finger" },
	{ 0x14, "Ring Finger" },
	{ 0x15, "Little Finger" },
	{ 0x16, "Head" },
	{ 0x17, "Shoulder" },
	{ 0x18, "Hip" },
	{ 0x19, "Waist" },
	{ 0x1a, "Thigh" },
	{ 0x1b, "Knee" },
	{ 0x1c, "calf" },
	{ 0x1d, "Ankle" },
	{ 0x1e, "Foot" },
	{ 0x1f, "Heel" },
	{ 0x20, "Ball of Foot" },
	{ 0x21, "Big Toe" },
	{ 0x22, "Second Toe" },
	{ 0x23, "Third Toe" },
	{ 0x24, "Fourth Toe" },
	{ 0x25, "Fifth Toe" },
	{ 0x26, "Brow" },
	{ 0x27, "Cheek" },
	{ 0x00, NULL },
	// clang-format on
};

/* HUT usage specs are represented as:
 *	(huttype << 16) + hutus
 */
static const struct genericstrtable hutus[] = {
	{ (0x01 << 16) + 0x000, "Undefined" },
	{ (0x01 << 16) + 0x001, "Pointer" },
	{ (0x01 << 16) + 0x002, "Mouse" },
	{ (0x01 << 16) + 0x004, "Joystick" },
	{ (0x01 << 16) + 0x005, "Gamepad" },
	{ (0x01 << 16) + 0x006, "Keyboard" },
	{ (0x01 << 16) + 0x007, "Keypad" },
	{ (0x01 << 16) + 0x008, "Multi-Axis Controller" },
	{ (0x01 << 16) + 0x030, "Direction-X" },
	{ (0x01 << 16) + 0x031, "Direction-Y" },
	{ (0x01 << 16) + 0x032, "Direction-Z" },
	{ (0x01 << 16) + 0x033, "Rotate-X" },
	{ (0x01 << 16) + 0x034, "Rotate-Y" },
	{ (0x01 << 16) + 0x035, "Rotate-Z" },
	{ (0x01 << 16) + 0x036, "Slider" },
	{ (0x01 << 16) + 0x037, "Dial" },
	{ (0x01 << 16) + 0x038, "Wheel" },
	{ (0x01 << 16) + 0x039, "Hat Switch" },
	{ (0x01 << 16) + 0x03a, "Counted Buffer" },
	{ (0x01 << 16) + 0x03b, "Byte Count" },
	{ (0x01 << 16) + 0x03c, "Motion Wakeup" },
	{ (0x01 << 16) + 0x03d, "Start" },
	{ (0x01 << 16) + 0x03e, "Select" },
	{ (0x01 << 16) + 0x040, "Vector-X" },
	{ (0x01 << 16) + 0x041, "Vector-Y" },
	{ (0x01 << 16) + 0x042, "Vector-Z" },
	{ (0x01 << 16) + 0x043, "Vector-X relative Body" },
	{ (0x01 << 16) + 0x044, "Vector-Y relative Body" },
	{ (0x01 << 16) + 0x045, "Vector-Z relative Body" },
	{ (0x01 << 16) + 0x046, "Vector" },
	{ (0x01 << 16) + 0x080, "System Control" },
	{ (0x01 << 16) + 0x081, "System Power Down" },
	{ (0x01 << 16) + 0x082, "System Sleep" },
	{ (0x01 << 16) + 0x083, "System Wake Up" },
	{ (0x01 << 16) + 0x084, "System Context Menu" },
	{ (0x01 << 16) + 0x085, "System Main Menu" },
	{ (0x01 << 16) + 0x086, "System App Menu" },
	{ (0x01 << 16) + 0x087, "System Menu Help" },
	{ (0x01 << 16) + 0x088, "System Menu Exit" },
	{ (0x01 << 16) + 0x089, "System Menu Select" },
	{ (0x01 << 16) + 0x08a, "System Menu Right" },
	{ (0x01 << 16) + 0x08b, "System Menu Left" },
	{ (0x01 << 16) + 0x08c, "System Menu Up" },
	{ (0x01 << 16) + 0x08d, "System Menu Down" },
	{ (0x01 << 16) + 0x090, "Direction Pad Up" },
	{ (0x01 << 16) + 0x091, "Direction Pad Down" },
	{ (0x01 << 16) + 0x092, "Direction Pad Right" },
	{ (0x01 << 16) + 0x093, "Direction Pad Left" },
	{ (0x02 << 16) + 0x000, "Undefined" },
	{ (0x02 << 16) + 0x001, "Flight Simulation Device" },
	{ (0x02 << 16) + 0x002, "Automobile Simulation Device" },
	{ (0x02 << 16) + 0x003, "Tank Simulation Device" },
	{ (0x02 << 16) + 0x004, "Spaceship Simulation Device" },
	{ (0x02 << 16) + 0x005, "Submarine Simulation Device" },
	{ (0x02 << 16) + 0x006, "Sailing Simulation Device" },
	{ (0x02 << 16) + 0x007, "Motorcycle Simulation Device" },
	{ (0x02 << 16) + 0x008, "Sports Simulation Device" },
	{ (0x02 << 16) + 0x009, "Airplane Simulation Device" },
	{ (0x02 << 16) + 0x00a, "Helicopter Simulation Device" },
	{ (0x02 << 16) + 0x00b, "Magic Carpet Simulation Device" },
	{ (0x02 << 16) + 0x00c, "Bicycle Simulation Device" },
	{ (0x02 << 16) + 0x020, "Flight Control Stick" },
	{ (0x02 << 16) + 0x021, "Flight Stick" },
	{ (0x02 << 16) + 0x022, "Cyclic Control" },
	{ (0x02 << 16) + 0x023, "Cyclic Trim" },
	{ (0x02 << 16) + 0x024, "Flight Yoke" },
	{ (0x02 << 16) + 0x025, "Track Control" },
	{ (0x02 << 16) + 0x0b0, "Aileron" },
	{ (0x02 << 16) + 0x0b1, "Aileron Trim" },
	{ (0x02 << 16) + 0x0b2, "Anti-Torque Control" },
	{ (0x02 << 16) + 0x0b3, "Autopilot Enable" },
	{ (0x02 << 16) + 0x0b4, "Chaff Release" },
	{ (0x02 << 16) + 0x0b5, "Collective Control" },
	{ (0x02 << 16) + 0x0b6, "Dive Break" },
	{ (0x02 << 16) + 0x0b7, "Electronic Countermeasures" },
	{ (0x02 << 16) + 0x0b8, "Elevator" },
	{ (0x02 << 16) + 0x0b9, "Elevator Trim" },
	{ (0x02 << 16) + 0x0ba, "Rudder" },
	{ (0x02 << 16) + 0x0bb, "Throttle" },
	{ (0x02 << 16) + 0x0bc, "Flight COmmunications" },
	{ (0x02 << 16) + 0x0bd, "Flare Release" },
	{ (0x02 << 16) + 0x0be, "Landing Gear" },
	{ (0x02 << 16) + 0x0bf, "Toe Break" },
	{ (0x02 << 16) + 0x0c0, "Trigger" },
	{ (0x02 << 16) + 0x0c1, "Weapon Arm" },
	{ (0x02 << 16) + 0x0c2, "Weapons Select" },
	{ (0x02 << 16) + 0x0c3, "Wing Flaps" },
	{ (0x02 << 16) + 0x0c4, "Accelerator" },
	{ (0x02 << 16) + 0x0c5, "Brake" },
	{ (0x02 << 16) + 0x0c6, "Clutch" },
	{ (0x02 << 16) + 0x0c7, "Shifter" },
	{ (0x02 << 16) + 0x0c8, "Steering" },
	{ (0x02 << 16) + 0x0c9, "Turret Direction" },
	{ (0x02 << 16) + 0x0ca, "Barrel Elevation" },
	{ (0x02 << 16) + 0x0cb, "Drive Plane" },
	{ (0x02 << 16) + 0x0cc, "Ballast" },
	{ (0x02 << 16) + 0x0cd, "Bicylce Crank" },
	{ (0x02 << 16) + 0x0ce, "Handle Bars" },
	{ (0x02 << 16) + 0x0cf, "Front Brake" },
	{ (0x02 << 16) + 0x0d0, "Rear Brake" },
	{ (0x03 << 16) + 0x000, "Unidentified" },
	{ (0x03 << 16) + 0x001, "Belt" },
	{ (0x03 << 16) + 0x002, "Body Suit" },
	{ (0x03 << 16) + 0x003, "Flexor" },
	{ (0x03 << 16) + 0x004, "Glove" },
	{ (0x03 << 16) + 0x005, "Head Tracker" },
	{ (0x03 << 16) + 0x006, "Head Mounted Display" },
	{ (0x03 << 16) + 0x007, "Hand Tracker" },
	{ (0x03 << 16) + 0x008, "Oculometer" },
	{ (0x03 << 16) + 0x009, "Vest" },
	{ (0x03 << 16) + 0x00a, "Animatronic Device" },
	{ (0x03 << 16) + 0x020, "Stereo Enable" },
	{ (0x03 << 16) + 0x021, "Display Enable" },
	{ (0x04 << 16) + 0x000, "Unidentified" },
	{ (0x04 << 16) + 0x001, "Baseball Bat" },
	{ (0x04 << 16) + 0x002, "Golf Club" },
	{ (0x04 << 16) + 0x003, "Rowing Machine" },
	{ (0x04 << 16) + 0x004, "Treadmill" },
	{ (0x04 << 16) + 0x030, "Oar" },
	{ (0x04 << 16) + 0x031, "Slope" },
	{ (0x04 << 16) + 0x032, "Rate" },
	{ (0x04 << 16) + 0x033, "Stick Speed" },
	{ (0x04 << 16) + 0x034, "Stick Face Angle" },
	{ (0x04 << 16) + 0x035, "Stick Heel/Toe" },
	{ (0x04 << 16) + 0x036, "Stick Follow Through" },
	{ (0x04 << 16) + 0x038, "Stick Type" },
	{ (0x04 << 16) + 0x039, "Stick Height" },
	{ (0x04 << 16) + 0x047, "Stick Temp" },
	{ (0x04 << 16) + 0x050, "Putter" },
	{ (0x04 << 16) + 0x051, "1 Iron" },
	{ (0x04 << 16) + 0x052, "2 Iron" },
	{ (0x04 << 16) + 0x053, "3 Iron" },
	{ (0x04 << 16) + 0x054, "4 Iron" },
	{ (0x04 << 16) + 0x055, "5 Iron" },
	{ (0x04 << 16) + 0x056, "6 Iron" },
	{ (0x04 << 16) + 0x057, "7 Iron" },
	{ (0x04 << 16) + 0x058, "8 Iron" },
	{ (0x04 << 16) + 0x059, "9 Iron" },
	{ (0x04 << 16) + 0x05a, "10 Iron" },
	{ (0x04 << 16) + 0x05b, "11 Iron" },
	{ (0x04 << 16) + 0x05c, "Sand Wedge" },
	{ (0x04 << 16) + 0x05d, "Loft Wedge" },
	{ (0x04 << 16) + 0x05e, "Power Wedge" },
	{ (0x04 << 16) + 0x05f, "1 Wood" },
	{ (0x04 << 16) + 0x060, "3 Wood" },
	{ (0x04 << 16) + 0x061, "5 Wood" },
	{ (0x04 << 16) + 0x062, "7 Wood" },
	{ (0x04 << 16) + 0x063, "9 Wood" },
	{ (0x05 << 16) + 0x000, "Undefined" },
	{ (0x05 << 16) + 0x001, "3D Game Controller" },
	{ (0x05 << 16) + 0x002, "Pinball Device" },
	{ (0x05 << 16) + 0x003, "Gun Device" },
	{ (0x05 << 16) + 0x020, "Point Of View" },
	{ (0x05 << 16) + 0x021, "Turn Right/Left" },
	{ (0x05 << 16) + 0x022, "Pitch Right/Left" },
	{ (0x05 << 16) + 0x023, "Roll Forward/Backward" },
	{ (0x05 << 16) + 0x024, "Move Right/Left" },
	{ (0x05 << 16) + 0x025, "Move Forward/Backward" },
	{ (0x05 << 16) + 0x026, "Move Up/Down" },
	{ (0x05 << 16) + 0x027, "Lean Right/Left" },
	{ (0x05 << 16) + 0x028, "Lean Forward/Backward" },
	{ (0x05 << 16) + 0x029, "Height of POV" },
	{ (0x05 << 16) + 0x02a, "Flipper" },
	{ (0x05 << 16) + 0x02b, "Secondary Flipper" },
	{ (0x05 << 16) + 0x02c, "Bump" },
	{ (0x05 << 16) + 0x02d, "New Game" },
	{ (0x05 << 16) + 0x02e, "Shoot Ball" },
	{ (0x05 << 16) + 0x02f, "Player" },
	{ (0x05 << 16) + 0x030, "Gun Bolt" },
	{ (0x05 << 16) + 0x031, "Gun Clip" },
	{ (0x05 << 16) + 0x032, "Gun Selector" },
	{ (0x05 << 16) + 0x033, "Gun Single Shot" },
	{ (0x05 << 16) + 0x034, "Gun Burst" },
	{ (0x05 << 16) + 0x035, "Gun Automatic" },
	{ (0x05 << 16) + 0x036, "Gun Safety" },
	{ (0x05 << 16) + 0x037, "Gamepad Fire/Jump" },
	{ (0x05 << 16) + 0x038, "Gamepad Fun" },
	{ (0x05 << 16) + 0x039, "Gamepad Trigger" },
	{ (0x07 << 16) + 0x000, "No Event" },
	{ (0x07 << 16) + 0x001, "Keyboard ErrorRollOver" },
	{ (0x07 << 16) + 0x002, "Keyboard POSTfail" },
	{ (0x07 << 16) + 0x003, "Keyboard Error Undefined" },
	{ (0x07 << 16) + 0x004, "A" },
	{ (0x07 << 16) + 0x005, "B" },
	{ (0x07 << 16) + 0x006, "C" },
	{ (0x07 << 16) + 0x007, "D" },
	{ (0x07 << 16) + 0x008, "E" },
	{ (0x07 << 16) + 0x009, "F" },
	{ (0x07 << 16) + 0x00a, "G" },
	{ (0x07 << 16) + 0x00b, "H" },
	{ (0x07 << 16) + 0x00c, "I" },
	{ (0x07 << 16) + 0x00d, "J" },
	{ (0x07 << 16) + 0x00e, "K" },
	{ (0x07 << 16) + 0x00f, "L" },
	{ (0x07 << 16) + 0x010, "M" },
	{ (0x07 << 16) + 0x011, "N" },
	{ (0x07 << 16) + 0x012, "O" },
	{ (0x07 << 16) + 0x013, "P" },
	{ (0x07 << 16) + 0x014, "Q" },
	{ (0x07 << 16) + 0x015, "R" },
	{ (0x07 << 16) + 0x016, "S" },
	{ (0x07 << 16) + 0x017, "T" },
	{ (0x07 << 16) + 0x018, "U" },
	{ (0x07 << 16) + 0x019, "V" },
	{ (0x07 << 16) + 0x01a, "W" },
	{ (0x07 << 16) + 0x01b, "X" },
	{ (0x07 << 16) + 0x01c, "Y" },
	{ (0x07 << 16) + 0x01d, "Z" },
	{ (0x07 << 16) + 0x01e, "1 and ! (One and Exclamation)" },
	{ (0x07 << 16) + 0x01f, "2 and @ (2 and at)" },
	{ (0x07 << 16) + 0x020, "3 and # (3 and Hash)" },
	{ (0x07 << 16) + 0x021, "4 and $ (4 and Dollar Sign)" },
	{ (0x07 << 16) + 0x022, "5 and % (5 and Percent Sign)" },
	{ (0x07 << 16) + 0x023, "6 and ^ (6 and circumflex)" },
	{ (0x07 << 16) + 0x024, "7 and & (Seven and Ampersand)" },
	{ (0x07 << 16) + 0x025, "8 and * (Eight and asterisk)" },
	{ (0x07 << 16) + 0x026, "9 and ( (Nine and Parenthesis Left)" },
	{ (0x07 << 16) + 0x027, "0 and ) (Zero and Parenthesis Right)" },
	{ (0x07 << 16) + 0x028, "Return (Enter)" },
	{ (0x07 << 16) + 0x029, "Escape" },
	{ (0x07 << 16) + 0x02a, "Delete (Backspace)" },
	{ (0x07 << 16) + 0x02b, "Tab" },
	{ (0x07 << 16) + 0x02c, "Space Bar" },
	{ (0x07 << 16) + 0x02d, "- and _ (Minus and underscore)" },
	{ (0x07 << 16) + 0x02e, "= and + (Equal and Plus)" },
	{ (0x07 << 16) + 0x02f, "[ and { (Bracket and Braces Left)" },
	{ (0x07 << 16) + 0x030, "] and } (Bracket and Braces Right)" },
	{ (0x07 << 16) + 0x031, "\\ and | (Backslash and Bar)" },
	{ (0x07 << 16) + 0x032, "# and ~ (Hash and Tilde, Non-US Keyboard near right shift)" },
	{ (0x07 << 16) + 0x033, "; and : (Semicolon and Colon)" },
	{ (0x07 << 16) + 0x034, "´ and \" (Accent Acute and Double Quotes)" },
	{ (0x07 << 16) + 0x035, "` and ~ (Accent Grace and Tilde)" },
	{ (0x07 << 16) + 0x036, ", and < (Comma and Less)" },
	{ (0x07 << 16) + 0x037, ". and > (Period and Greater)" },
	{ (0x07 << 16) + 0x038, "/ and ? (Slash and Question Mark)" },
	{ (0x07 << 16) + 0x039, "Caps Lock" },
	{ (0x07 << 16) + 0x03a, "F1" },
	{ (0x07 << 16) + 0x03b, "F2" },
	{ (0x07 << 16) + 0x03c, "F3" },
	{ (0x07 << 16) + 0x03d, "F4" },
	{ (0x07 << 16) + 0x03e, "F5" },
	{ (0x07 << 16) + 0x03f, "F6" },
	{ (0x07 << 16) + 0x040, "F7" },
	{ (0x07 << 16) + 0x041, "F8" },
	{ (0x07 << 16) + 0x042, "F9" },
	{ (0x07 << 16) + 0x043, "F10" },
	{ (0x07 << 16) + 0x044, "F11" },
	{ (0x07 << 16) + 0x045, "F12" },
	{ (0x07 << 16) + 0x046, "Print Screen" },
	{ (0x07 << 16) + 0x047, "Scroll Lock" },
	{ (0x07 << 16) + 0x048, "Pause" },
	{ (0x07 << 16) + 0x049, "Insert" },
	{ (0x07 << 16) + 0x04a, "Home" },
	{ (0x07 << 16) + 0x04b, "Page Up" },
	{ (0x07 << 16) + 0x04c, "Delete Forward (without Changing Position)" },
	{ (0x07 << 16) + 0x04d, "End" },
	{ (0x07 << 16) + 0x04e, "Page Down" },
	{ (0x07 << 16) + 0x04f, "Right Arrow" },
	{ (0x07 << 16) + 0x050, "Left Arrow" },
	{ (0x07 << 16) + 0x051, "Down Arrow" },
	{ (0x07 << 16) + 0x052, "Up Arrow" },
	{ (0x07 << 16) + 0x053, "Num Lock and Clear" },
	{ (0x07 << 16) + 0x054, "Keypad / (Division Sign)" },
	{ (0x07 << 16) + 0x055, "Keypad * (Multiplication Sign)" },
	{ (0x07 << 16) + 0x056, "Keypad - (Subtraction Sign)" },
	{ (0x07 << 16) + 0x057, "Keypad + (Addition Sign)" },
	{ (0x07 << 16) + 0x058, "Keypad Enter" },
	{ (0x07 << 16) + 0x059, "Keypad 1 and END" },
	{ (0x07 << 16) + 0x05a, "Keypad 2 and Down Arrow" },
	{ (0x07 << 16) + 0x05b, "Keypad 3 and Page Down" },
	{ (0x07 << 16) + 0x05c, "Keypad 4 and Left Arrow" },
	{ (0x07 << 16) + 0x05d, "Keypad 5 (Tactilei Raised)" },
	{ (0x07 << 16) + 0x05f, "Keypad 6 and Right Arrow" },
	{ (0x07 << 16) + 0x060, "Keypad 7 and Home" },
	{ (0x07 << 16) + 0x061, "Keypad 8 and Up Arrow" },
	{ (0x07 << 16) + 0x062, "Keypad 8 and Page Up" },
	{ (0x07 << 16) + 0x063, "Keypad . (decimal delimiter) and Delete" },
	{ (0x07 << 16) + 0x064, "\\ and | (Backslash and Bar, UK and Non-US Keyboard near left shift)" },
	{ (0x07 << 16) + 0x065, "Keyboard Application (Windows Key for Win95 or Compose)" },
	{ (0x07 << 16) + 0x066, "Power (not a key)" },
	{ (0x07 << 16) + 0x067, "Keypad = (Equal Sign)" },
	{ (0x07 << 16) + 0x068, "F13" },
	{ (0x07 << 16) + 0x069, "F14" },
	{ (0x07 << 16) + 0x06a, "F15" },
	{ (0x07 << 16) + 0x06b, "F16" },
	{ (0x07 << 16) + 0x06c, "F17" },
	{ (0x07 << 16) + 0x06d, "F18" },
	{ (0x07 << 16) + 0x06e, "F19" },
	{ (0x07 << 16) + 0x06f, "F20" },
	{ (0x07 << 16) + 0x070, "F21" },
	{ (0x07 << 16) + 0x071, "F22" },
	{ (0x07 << 16) + 0x072, "F23" },
	{ (0x07 << 16) + 0x073, "F24" },
	{ (0x07 << 16) + 0x074, "Execute" },
	{ (0x07 << 16) + 0x075, "Help" },
	{ (0x07 << 16) + 0x076, "Menu" },
	{ (0x07 << 16) + 0x077, "Select" },
	{ (0x07 << 16) + 0x078, "Stop" },
	{ (0x07 << 16) + 0x079, "Again" },
	{ (0x07 << 16) + 0x07a, "Undo" },
	{ (0x07 << 16) + 0x07b, "Cut" },
	{ (0x07 << 16) + 0x07c, "Copy" },
	{ (0x07 << 16) + 0x07d, "Paste" },
	{ (0x07 << 16) + 0x07e, "Find" },
	{ (0x07 << 16) + 0x07f, "Mute" },
	{ (0x07 << 16) + 0x080, "Volume Up" },
	{ (0x07 << 16) + 0x081, "Volume Down" },
	{ (0x07 << 16) + 0x082, "Locking Caps Lock" },
	{ (0x07 << 16) + 0x083, "Locking Num Lock" },
	{ (0x07 << 16) + 0x084, "Locking Scroll Lock" },
	{ (0x07 << 16) + 0x085, "Keypad Comma" },
	{ (0x07 << 16) + 0x086, "Keypad Equal Sign (AS/400)" },
	{ (0x07 << 16) + 0x087, "International 1 (PC98)" },
	{ (0x07 << 16) + 0x088, "International 2 (PC98)" },
	{ (0x07 << 16) + 0x089, "International 3 (PC98)" },
	{ (0x07 << 16) + 0x08a, "International 4 (PC98)" },
	{ (0x07 << 16) + 0x08b, "International 5 (PC98)" },
	{ (0x07 << 16) + 0x08c, "International 6 (PC98)" },
	{ (0x07 << 16) + 0x08d, "International 7 (Toggle Single/Double Byte Mode)" },
	{ (0x07 << 16) + 0x08e, "International 8" },
	{ (0x07 << 16) + 0x08f, "International 9" },
	{ (0x07 << 16) + 0x090, "LANG 1 (Hangul/English Toggle, Korea)" },
	{ (0x07 << 16) + 0x091, "LANG 2 (Hanja Conversion, Korea)" },
	{ (0x07 << 16) + 0x092, "LANG 3 (Katakana, Japan)" },
	{ (0x07 << 16) + 0x093, "LANG 4 (Hiragana, Japan)" },
	{ (0x07 << 16) + 0x094, "LANG 5 (Zenkaku/Hankaku, Japan)" },
	{ (0x07 << 16) + 0x095, "LANG 6" },
	{ (0x07 << 16) + 0x096, "LANG 7" },
	{ (0x07 << 16) + 0x097, "LANG 8" },
	{ (0x07 << 16) + 0x098, "LANG 9" },
	{ (0x07 << 16) + 0x099, "Alternate Erase" },
	{ (0x07 << 16) + 0x09a, "SysReq/Attention" },
	{ (0x07 << 16) + 0x09b, "Cancel" },
	{ (0x07 << 16) + 0x09c, "Clear" },
	{ (0x07 << 16) + 0x09d, "Prior" },
	{ (0x07 << 16) + 0x09e, "Return" },
	{ (0x07 << 16) + 0x09f, "Separator" },
	{ (0x07 << 16) + 0x0a0, "Out" },
	{ (0x07 << 16) + 0x0a1, "Open" },
	{ (0x07 << 16) + 0x0a2, "Clear/Again" },
	{ (0x07 << 16) + 0x0a3, "CrSel/Props" },
	{ (0x07 << 16) + 0x0a4, "ExSel" },
	{ (0x07 << 16) + 0x0e0, "Control Left" },
	{ (0x07 << 16) + 0x0e1, "Shift Left" },
	{ (0x07 << 16) + 0x0e2, "Alt Left" },
	{ (0x07 << 16) + 0x0e3, "GUI Left" },
	{ (0x07 << 16) + 0x0e4, "Control Right" },
	{ (0x07 << 16) + 0x0e5, "Shift Right" },
	{ (0x07 << 16) + 0x0e6, "Alt Right" },
	{ (0x07 << 16) + 0x0e7, "GUI Right" },
	{ (0x08 << 16) + 0x000, "Undefined" },
	{ (0x08 << 16) + 0x001, "NumLock" },
	{ (0x08 << 16) + 0x002, "CapsLock" },
	{ (0x08 << 16) + 0x003, "Scroll Lock" },
	{ (0x08 << 16) + 0x004, "Compose" },
	{ (0x08 << 16) + 0x005, "Kana" },
	{ (0x08 << 16) + 0x006, "Power" },
	{ (0x08 << 16) + 0x007, "Shift" },
	{ (0x08 << 16) + 0x008, "Do not disturb" },
	{ (0x08 << 16) + 0x009, "Mute" },
	{ (0x08 << 16) + 0x00a, "Tone Enabke" },
	{ (0x08 << 16) + 0x00b, "High Cut Filter" },
	{ (0x08 << 16) + 0x00c, "Low Cut Filter" },
	{ (0x08 << 16) + 0x00d, "Equalizer Enable" },
	{ (0x08 << 16) + 0x00e, "Sound Field ON" },
	{ (0x08 << 16) + 0x00f, "Surround On" },
	{ (0x08 << 16) + 0x010, "Repeat" },
	{ (0x08 << 16) + 0x011, "Stereo" },
	{ (0x08 << 16) + 0x012, "Sampling Rate Detect" },
	{ (0x08 << 16) + 0x013, "Spinning" },
	{ (0x08 << 16) + 0x014, "CAV" },
	{ (0x08 << 16) + 0x015, "CLV" },
	{ (0x08 << 16) + 0x016, "Recording Format Detect" },
	{ (0x08 << 16) + 0x017, "Off-Hook" },
	{ (0x08 << 16) + 0x018, "Ring" },
	{ (0x08 << 16) + 0x019, "Message Waiting" },
	{ (0x08 << 16) + 0x01a, "Data Mode" },
	{ (0x08 << 16) + 0x01b, "Battery Operation" },
	{ (0x08 << 16) + 0x01c, "Battery OK" },
	{ (0x08 << 16) + 0x01d, "Battery Low" },
	{ (0x08 << 16) + 0x01e, "Speaker" },
	{ (0x08 << 16) + 0x01f, "Head Set" },
	{ (0x08 << 16) + 0x020, "Hold" },
	{ (0x08 << 16) + 0x021, "Microphone" },
	{ (0x08 << 16) + 0x022, "Coverage" },
	{ (0x08 << 16) + 0x023, "Night Mode" },
	{ (0x08 << 16) + 0x024, "Send Calls" },
	{ (0x08 << 16) + 0x025, "Call Pickup" },
	{ (0x08 << 16) + 0x026, "Conference" },
	{ (0x08 << 16) + 0x027, "Stand-by" },
	{ (0x08 << 16) + 0x028, "Camera On" },
	{ (0x08 << 16) + 0x029, "Camera Off" },
	{ (0x08 << 16) + 0x02a, "On-Line" },
	{ (0x08 << 16) + 0x02b, "Off-Line" },
	{ (0x08 << 16) + 0x02c, "Busy" },
	{ (0x08 << 16) + 0x02d, "Ready" },
	{ (0x08 << 16) + 0x02e, "Paper-Out" },
	{ (0x08 << 16) + 0x02f, "Paper-Jam" },
	{ (0x08 << 16) + 0x030, "Remote" },
	{ (0x08 << 16) + 0x031, "Forward" },
	{ (0x08 << 16) + 0x032, "Reverse" },
	{ (0x08 << 16) + 0x033, "Stop" },
	{ (0x08 << 16) + 0x034, "Rewind" },
	{ (0x08 << 16) + 0x035, "Fast Forward" },
	{ (0x08 << 16) + 0x036, "Play" },
	{ (0x08 << 16) + 0x037, "Pause" },
	{ (0x08 << 16) + 0x038, "Record" },
	{ (0x08 << 16) + 0x039, "Error" },
	{ (0x08 << 16) + 0x03a, "Usage Selected Indicator" },
	{ (0x08 << 16) + 0x03b, "Usage In Use Indicator" },
	{ (0x08 << 16) + 0x03c, "Usage Multi Indicator" },
	{ (0x08 << 16) + 0x03d, "Indicator On" },
	{ (0x08 << 16) + 0x03e, "Indicator Flash" },
	{ (0x08 << 16) + 0x03f, "Indicator Slow Blink" },
	{ (0x08 << 16) + 0x040, "Indicator Fast Blink" },
	{ (0x08 << 16) + 0x041, "Indicator Off" },
	{ (0x08 << 16) + 0x042, "Flash On Time" },
	{ (0x08 << 16) + 0x043, "Slow Blink On Time" },
	{ (0x08 << 16) + 0x044, "Slow Blink Off Time" },
	{ (0x08 << 16) + 0x045, "Fast Blink On Time" },
	{ (0x08 << 16) + 0x046, "Fast Blink Off Time" },
	{ (0x08 << 16) + 0x047, "Usage Color Indicator" },
	{ (0x08 << 16) + 0x048, "Indicator Red" },
	{ (0x08 << 16) + 0x049, "Indicator Green" },
	{ (0x08 << 16) + 0x04a, "Indicator Amber" },
	{ (0x08 << 16) + 0x04b, "Generic Indicator" },
	{ (0x08 << 16) + 0x04c, "System Suspend" },
	{ (0x08 << 16) + 0x04d, "External Power Connected" },
	{ (0x09 << 16) + 0x000, "No Button Pressed" },
	{ (0x09 << 16) + 0x001, "Button 1 (Primary)" },
	{ (0x09 << 16) + 0x002, "Button 2 (Secondary)" },
	{ (0x09 << 16) + 0x003, "Button 3 (Tertiary)" },
	{ (0x09 << 16) + 0x004, "Button 4" },
	{ (0x09 << 16) + 0x005, "Button 5" },
	{ (0x0a << 16) + 0x001, "Instance 1" },
	{ (0x0a << 16) + 0x002, "Instance 2" },
	{ (0x0a << 16) + 0x003, "Instance 3" },
	{ (0x0b << 16) + 0x000, "Unassigned" },
	{ (0x0b << 16) + 0x001, "Phone" },
	{ (0x0b << 16) + 0x002, "Answering Machine" },
	{ (0x0b << 16) + 0x003, "Message Controls" },
	{ (0x0b << 16) + 0x004, "Handset" },
	{ (0x0b << 16) + 0x005, "Headset" },
	{ (0x0b << 16) + 0x006, "Telephony Key Pad" },
	{ (0x0b << 16) + 0x007, "Programmable Button" },
	{ (0x0b << 16) + 0x020, "Hook Switch" },
	{ (0x0b << 16) + 0x021, "Flash" },
	{ (0x0b << 16) + 0x022, "Feature" },
	{ (0x0b << 16) + 0x023, "Hold" },
	{ (0x0b << 16) + 0x024, "Redial" },
	{ (0x0b << 16) + 0x025, "Transfer" },
	{ (0x0b << 16) + 0x026, "Drop" },
	{ (0x0b << 16) + 0x027, "Park" },
	{ (0x0b << 16) + 0x028, "Forward Calls" },
	{ (0x0b << 16) + 0x029, "Alternate Function" },
	{ (0x0b << 16) + 0x02a, "Line" },
	{ (0x0b << 16) + 0x02b, "Speaker Phone" },
	{ (0x0b << 16) + 0x02c, "Conference" },
	{ (0x0b << 16) + 0x02d, "Ring Enable" },
	{ (0x0b << 16) + 0x02e, "Ring Select" },
	{ (0x0b << 16) + 0x02f, "Phone Mute" },
	{ (0x0b << 16) + 0x030, "Caller ID" },
	{ (0x0b << 16) + 0x050, "Speed Dial" },
	{ (0x0b << 16) + 0x051, "Store Number" },
	{ (0x0b << 16) + 0x052, "Recall Number" },
	{ (0x0b << 16) + 0x053, "Phone Directory" },
	{ (0x0b << 16) + 0x070, "Voice Mail" },
	{ (0x0b << 16) + 0x071, "Screen Calls" },
	{ (0x0b << 16) + 0x072, "Do Not Disturb" },
	{ (0x0b << 16) + 0x073, "Message" },
	{ (0x0b << 16) + 0x074, "Answer On/Offf" },
	{ (0x0b << 16) + 0x090, "Inside Dial Tone" },
	{ (0x0b << 16) + 0x091, "Outside Dial Tone" },
	{ (0x0b << 16) + 0x092, "Inside Ring Tone" },
	{ (0x0b << 16) + 0x093, "Outside Ring Tone" },
	{ (0x0b << 16) + 0x094, "Priority Ring Tone" },
	{ (0x0b << 16) + 0x095, "Inside Ringback" },
	{ (0x0b << 16) + 0x096, "Priority Ringback" },
	{ (0x0b << 16) + 0x097, "Line Busy Tone" },
	{ (0x0b << 16) + 0x098, "Recorder Tone" },
	{ (0x0b << 16) + 0x099, "Call Waiting Tone" },
	{ (0x0b << 16) + 0x09a, "Confirmation Tone 1" },
	{ (0x0b << 16) + 0x09b, "Confirmation Tone 2" },
	{ (0x0b << 16) + 0x09c, "Tones Off" },
	{ (0x0b << 16) + 0x09d, "Outside Ringback" },
	{ (0x0b << 16) + 0x0b0, "Key 1" },
	{ (0x0b << 16) + 0x0b1, "Key 2" },
	{ (0x0b << 16) + 0x0b3, "Key 3" },
	{ (0x0b << 16) + 0x0b4, "Key 4" },
	{ (0x0b << 16) + 0x0b5, "Key 5" },
	{ (0x0b << 16) + 0x0b6, "Key 6" },
	{ (0x0b << 16) + 0x0b7, "Key 7" },
	{ (0x0b << 16) + 0x0b8, "Key 8" },
	{ (0x0b << 16) + 0x0b9, "Key 9" },
	{ (0x0b << 16) + 0x0ba, "Key Star" },
	{ (0x0b << 16) + 0x0bb, "Key Pound" },
	{ (0x0b << 16) + 0x0bc, "Key A" },
	{ (0x0b << 16) + 0x0bd, "Key B" },
	{ (0x0b << 16) + 0x0be, "Key C" },
	{ (0x0b << 16) + 0x0bf, "Key D" },
	{ (0x0c << 16) + 0x000, "Unassigned" },
	{ (0x0c << 16) + 0x001, "Consumer Control" },
	{ (0x0c << 16) + 0x002, "Numeric Key Pad" },
	{ (0x0c << 16) + 0x003, "Programmable Buttons" },
	{ (0x0c << 16) + 0x020, "+10" },
	{ (0x0c << 16) + 0x021, "+100" },
	{ (0x0c << 16) + 0x022, "AM/PM" },
	{ (0x0c << 16) + 0x030, "Power" },
	{ (0x0c << 16) + 0x031, "Reset" },
	{ (0x0c << 16) + 0x032, "Sleep" },
	{ (0x0c << 16) + 0x033, "Sleep After" },
	{ (0x0c << 16) + 0x034, "Sleep Mode" },
	{ (0x0c << 16) + 0x035, "Illumination" },
	{ (0x0c << 16) + 0x036, "Function Buttons" },
	{ (0x0c << 16) + 0x040, "Menu" },
	{ (0x0c << 16) + 0x041, "Menu Pick" },
	{ (0x0c << 16) + 0x042, "Menu Up" },
	{ (0x0c << 16) + 0x043, "Menu Down" },
	{ (0x0c << 16) + 0x044, "Menu Left" },
	{ (0x0c << 16) + 0x045, "Menu Right" },
	{ (0x0c << 16) + 0x046, "Menu Escape" },
	{ (0x0c << 16) + 0x047, "Menu Value Increase" },
	{ (0x0c << 16) + 0x048, "Menu Value Decrease" },
	{ (0x0c << 16) + 0x060, "Data on Screen" },
	{ (0x0c << 16) + 0x061, "Closed Caption" },
	{ (0x0c << 16) + 0x062, "Closed Caption Select" },
	{ (0x0c << 16) + 0x063, "VCR/TV" },
	{ (0x0c << 16) + 0x064, "Broadcast Mode" },
	{ (0x0c << 16) + 0x065, "Snapshot" },
	{ (0x0c << 16) + 0x066, "Still" },
	{ (0x0c << 16) + 0x080, "Selection" },
	{ (0x0c << 16) + 0x081, "Assign Selection" },
	{ (0x0c << 16) + 0x082, "Mode Step" },
	{ (0x0c << 16) + 0x083, "Recall Last" },
	{ (0x0c << 16) + 0x084, "Enter Channel" },
	{ (0x0c << 16) + 0x085, "Order Movie" },
	{ (0x0c << 16) + 0x086, "Channel" },
	{ (0x0c << 16) + 0x087, "Media Selection" },
	{ (0x0c << 16) + 0x088, "Media Select Computer" },
	{ (0x0c << 16) + 0x089, "Media Select TV" },
	{ (0x0c << 16) + 0x08a, "Media Select WWW" },
	{ (0x0c << 16) + 0x08b, "Media Select DVD" },
	{ (0x0c << 16) + 0x08c, "Media Select Telephone" },
	{ (0x0c << 16) + 0x08d, "Media Select Program Guide" },
	{ (0x0c << 16) + 0x08e, "Media Select Video Phone" },
	{ (0x0c << 16) + 0x08f, "Media Select Games" },
	{ (0x0c << 16) + 0x090, "Media Select Messages" },
	{ (0x0c << 16) + 0x091, "Media Select CD" },
	{ (0x0c << 16) + 0x092, "Media Select VCR" },
	{ (0x0c << 16) + 0x093, "Media Select Tuner" },
	{ (0x0c << 16) + 0x094, "Quit" },
	{ (0x0c << 16) + 0x095, "Help" },
	{ (0x0c << 16) + 0x096, "Media Select Tape" },
	{ (0x0c << 16) + 0x097, "Media Select Cable" },
	{ (0x0c << 16) + 0x098, "Media Select Satellite" },
	{ (0x0c << 16) + 0x099, "Media Select Security" },
	{ (0x0c << 16) + 0x09a, "Media Select Home" },
	{ (0x0c << 16) + 0x09b, "Media Select Call" },
	{ (0x0c << 16) + 0x09c, "Channel Increment" },
	{ (0x0c << 16) + 0x09d, "Channel Decrement" },
	{ (0x0c << 16) + 0x09e, "Media Select SAP" },
	{ (0x0c << 16) + 0x0a0, "VCR Plus" },
	{ (0x0c << 16) + 0x0a1, "Once" },
	{ (0x0c << 16) + 0x0a2, "Daily" },
	{ (0x0c << 16) + 0x0a3, "Weekly" },
	{ (0x0c << 16) + 0x0a4, "Monthly" },
	{ (0x0c << 16) + 0x0b0, "Play" },
	{ (0x0c << 16) + 0x0b1, "Pause" },
	{ (0x0c << 16) + 0x0b2, "Record" },
	{ (0x0c << 16) + 0x0b3, "Fast Forward" },
	{ (0x0c << 16) + 0x0b4, "Rewind" },
	{ (0x0c << 16) + 0x0b5, "Scan Next Track" },
	{ (0x0c << 16) + 0x0b6, "Scan Previous Track" },
	{ (0x0c << 16) + 0x0b7, "Stop" },
	{ (0x0c << 16) + 0x0b8, "Eject" },
	{ (0x0c << 16) + 0x0b9, "Random Play" },
	{ (0x0c << 16) + 0x0ba, "Select Disc" },
	{ (0x0c << 16) + 0x0bb, "Enter Disc" },
	{ (0x0c << 16) + 0x0bc, "Repeat" },
	{ (0x0c << 16) + 0x0bd, "Tracking" },
	{ (0x0c << 16) + 0x0be, "Track Normal" },
	{ (0x0c << 16) + 0x0bf, "Slow Tracking" },
	{ (0x0c << 16) + 0x0c0, "Frame Forward" },
	{ (0x0c << 16) + 0x0c1, "Frame Back" },
	{ (0x0c << 16) + 0x0c2, "Mark" },
	{ (0x0c << 16) + 0x0c3, "Clear Mark" },
	{ (0x0c << 16) + 0x0c4, "Repeat from Mark" },
	{ (0x0c << 16) + 0x0c5, "Return to Mark" },
	{ (0x0c << 16) + 0x0c6, "Search Mark Forward" },
	{ (0x0c << 16) + 0x0c7, "Search Mark Backward" },
	{ (0x0c << 16) + 0x0c8, "Counter Reset" },
	{ (0x0c << 16) + 0x0c9, "Show Counter" },
	{ (0x0c << 16) + 0x0ca, "Tracking Increment" },
	{ (0x0c << 16) + 0x0cb, "Tracking Decrement" },
	{ (0x0c << 16) + 0x0cc, "Stop/Eject" },
	{ (0x0c << 16) + 0x0cd, "Play/Pause" },
	{ (0x0c << 16) + 0x0ce, "Play/Skip" },
	{ (0x0c << 16) + 0x0e0, "Volume" },
	{ (0x0c << 16) + 0x0e1, "Balance" },
	{ (0x0c << 16) + 0x0e2, "Mute" },
	{ (0x0c << 16) + 0x0e3, "Bass" },
	{ (0x0c << 16) + 0x0e4, "Treble" },
	{ (0x0c << 16) + 0x0e5, "Bass Boost" },
	{ (0x0c << 16) + 0x0e6, "Surround Mode" },
	{ (0x0c << 16) + 0x0e7, "Loudness" },
	{ (0x0c << 16) + 0x0e8, "MPX" },
	{ (0x0c << 16) + 0x0e9, "Volume Increment" },
	{ (0x0c << 16) + 0x0ea, "Volume Decrement" },
	{ (0x0c << 16) + 0x0f0, "Speed Select" },
	{ (0x0c << 16) + 0x0f1, "Playback Speed" },
	{ (0x0c << 16) + 0x0f2, "Standard Play" },
	{ (0x0c << 16) + 0x0f3, "Long Play" },
	{ (0x0c << 16) + 0x0f4, "Extended Play" },
	{ (0x0c << 16) + 0x0f5, "Slow" },
	{ (0x0c << 16) + 0x100, "Fan Enable" },
	{ (0x0c << 16) + 0x101, "Fan Speed" },
	{ (0x0c << 16) + 0x102, "Light Enable" },
	{ (0x0c << 16) + 0x103, "Light Illumination Level" },
	{ (0x0c << 16) + 0x104, "Climate Control Enable" },
	{ (0x0c << 16) + 0x105, "Room Temperature" },
	{ (0x0c << 16) + 0x106, "Security Enable" },
	{ (0x0c << 16) + 0x107, "Fire Alarm" },
	{ (0x0c << 16) + 0x108, "Police Alarm" },
	{ (0x0c << 16) + 0x150, "Balance Right" },
	{ (0x0c << 16) + 0x151, "Balance Left" },
	{ (0x0c << 16) + 0x152, "Bass Increment" },
	{ (0x0c << 16) + 0x153, "Bass Decrement" },
	{ (0x0c << 16) + 0x154, "Treble Increment" },
	{ (0x0c << 16) + 0x155, "Treble Decrement" },
	{ (0x0c << 16) + 0x160, "Speaker System" },
	{ (0x0c << 16) + 0x161, "Channel Left" },
	{ (0x0c << 16) + 0x162, "Channel Right" },
	{ (0x0c << 16) + 0x163, "Channel Center" },
	{ (0x0c << 16) + 0x164, "Channel Front" },
	{ (0x0c << 16) + 0x165, "Channel Center Front" },
	{ (0x0c << 16) + 0x166, "Channel Side" },
	{ (0x0c << 16) + 0x167, "Channel Surround" },
	{ (0x0c << 16) + 0x168, "Channel Low Frequency Enhancement" },
	{ (0x0c << 16) + 0x169, "Channel Top" },
	{ (0x0c << 16) + 0x16a, "Channel Unknown" },
	{ (0x0c << 16) + 0x170, "Sub-Channel" },
	{ (0x0c << 16) + 0x171, "Sub-Channel Increment" },
	{ (0x0c << 16) + 0x172, "Sub-Channel Decrement" },
	{ (0x0c << 16) + 0x173, "Alternative Audio Increment" },
	{ (0x0c << 16) + 0x174, "Alternative Audio Decrement" },
	{ (0x0c << 16) + 0x180, "Application Launch Buttons" },
	{ (0x0c << 16) + 0x181, "AL Launch Button Configuration Tool" },
	{ (0x0c << 16) + 0x182, "AL Launch Button Configuration" },
	{ (0x0c << 16) + 0x183, "AL Consumer Control Configuration" },
	{ (0x0c << 16) + 0x184, "AL Word Processor" },
	{ (0x0c << 16) + 0x185, "AL Text Editor" },
	{ (0x0c << 16) + 0x186, "AL Spreadsheet" },
	{ (0x0c << 16) + 0x187, "AL Graphics Editor" },
	{ (0x0c << 16) + 0x188, "AL Presentation App" },
	{ (0x0c << 16) + 0x189, "AL Database App" },
	{ (0x0c << 16) + 0x18a, "AL Email Reader" },
	{ (0x0c << 16) + 0x18b, "AL Newsreader" },
	{ (0x0c << 16) + 0x18c, "AL Voicemail" },
	{ (0x0c << 16) + 0x18d, "AL Contacts/Address Book" },
	{ (0x0c << 16) + 0x18e, "AL Calendar/Schedule" },
	{ (0x0c << 16) + 0x18f, "AL Task/Project Manager" },
	{ (0x0c << 16) + 0x190, "AL Log/Journal/Timecard" },
	{ (0x0c << 16) + 0x191, "AL Checkbook/Finance" },
	{ (0x0c << 16) + 0x192, "AL Calculator" },
	{ (0x0c << 16) + 0x193, "AL A/V Capture/Playback" },
	{ (0x0c << 16) + 0x194, "AL Local Machine Browser" },
	{ (0x0c << 16) + 0x195, "AL LAN/Wan Browser" },
	{ (0x0c << 16) + 0x196, "AL Internet Browser" },
	{ (0x0c << 16) + 0x197, "AL Remote Networking/ISP Connect" },
	{ (0x0c << 16) + 0x198, "AL Network Conference" },
	{ (0x0c << 16) + 0x199, "AL Network Chat" },
	{ (0x0c << 16) + 0x19a, "AL Telephony/Dialer" },
	{ (0x0c << 16) + 0x19b, "AL Logon" },
	{ (0x0c << 16) + 0x19c, "AL Logoff" },
	{ (0x0c << 16) + 0x19d, "AL Logon/Logoff" },
	{ (0x0c << 16) + 0x19e, "AL Terminal Local/Screensaver" },
	{ (0x0c << 16) + 0x19f, "AL Control Panel" },
	{ (0x0c << 16) + 0x1a0, "AL Command Line Processor/Run" },
	{ (0x0c << 16) + 0x1a1, "AL Process/Task Manager" },
	{ (0x0c << 16) + 0x1a2, "AL Select Task/Application" },
	{ (0x0c << 16) + 0x1a3, "AL Next Task/Application" },
	{ (0x0c << 16) + 0x1a4, "AL Previous Task/Application" },
	{ (0x0c << 16) + 0x1a5, "AL Preemptive Halt Task/Application" },
	{ (0x0c << 16) + 0x200, "Generic GUI Application Controls" },
	{ (0x0c << 16) + 0x201, "AC New" },
	{ (0x0c << 16) + 0x202, "AC Open" },
	{ (0x0c << 16) + 0x203, "AC CLose" },
	{ (0x0c << 16) + 0x204, "AC Exit" },
	{ (0x0c << 16) + 0x205, "AC Maximize" },
	{ (0x0c << 16) + 0x206, "AC Minimize" },
	{ (0x0c << 16) + 0x207, "AC Save" },
	{ (0x0c << 16) + 0x208, "AC Print" },
	{ (0x0c << 16) + 0x209, "AC Properties" },
	{ (0x0c << 16) + 0x21a, "AC Undo" },
	{ (0x0c << 16) + 0x21b, "AC Copy" },
	{ (0x0c << 16) + 0x21c, "AC Cut" },
	{ (0x0c << 16) + 0x21d, "AC Paste" },
	{ (0x0c << 16) + 0x21e, "AC Select All" },
	{ (0x0c << 16) + 0x21f, "AC Find" },
	{ (0x0c << 16) + 0x220, "AC Find and Replace" },
	{ (0x0c << 16) + 0x221, "AC Search" },
	{ (0x0c << 16) + 0x222, "AC Go To" },
	{ (0x0c << 16) + 0x223, "AC Home" },
	{ (0x0c << 16) + 0x224, "AC Back" },
	{ (0x0c << 16) + 0x225, "AC Forward" },
	{ (0x0c << 16) + 0x226, "AC Stop" },
	{ (0x0c << 16) + 0x227, "AC Refresh" },
	{ (0x0c << 16) + 0x228, "AC Previous Link" },
	{ (0x0c << 16) + 0x229, "AC Next Link" },
	{ (0x0c << 16) + 0x22b, "AC History" },
	{ (0x0c << 16) + 0x22c, "AC Subscriptions" },
	{ (0x0c << 16) + 0x22d, "AC Zoom In" },
	{ (0x0c << 16) + 0x22e, "AC Zoom Out" },
	{ (0x0c << 16) + 0x22f, "AC Zoom" },
	{ (0x0c << 16) + 0x230, "AC Full Screen View" },
	{ (0x0c << 16) + 0x231, "AC Normal View" },
	{ (0x0c << 16) + 0x232, "AC View Toggle" },
	{ (0x0c << 16) + 0x233, "AC Scroll Up" },
	{ (0x0c << 16) + 0x234, "AC Scroll Down" },
	{ (0x0c << 16) + 0x235, "AC Scroll" },
	{ (0x0c << 16) + 0x236, "AC Pan Left" },
	{ (0x0c << 16) + 0x237, "AC Pan Right" },
	{ (0x0c << 16) + 0x238, "AC Pan" },
	{ (0x0c << 16) + 0x239, "AC New Window" },
	{ (0x0c << 16) + 0x23a, "AC Tile Horizontally" },
	{ (0x0c << 16) + 0x23b, "AC Tile Vertically" },
	{ (0x0c << 16) + 0x23c, "AC Format" },
	{ (0x0d << 16) + 0x000, "Undefined" },
	{ (0x0d << 16) + 0x001, "Digitizer" },
	{ (0x0d << 16) + 0x002, "Pen" },
	{ (0x0d << 16) + 0x003, "Light Pen" },
	{ (0x0d << 16) + 0x004, "Touch Screen" },
	{ (0x0d << 16) + 0x005, "Touch Pad" },
	{ (0x0d << 16) + 0x006, "White Board" },
	{ (0x0d << 16) + 0x007, "Coordinate Measuring Machine" },
	{ (0x0d << 16) + 0x008, "3D Digitizer" },
	{ (0x0d << 16) + 0x009, "Stereo Plotter" },
	{ (0x0d << 16) + 0x00a, "Articulated Arm" },
	{ (0x0d << 16) + 0x00b, "Armature" },
	{ (0x0d << 16) + 0x00c, "Multiple Point Digitizer" },
	{ (0x0d << 16) + 0x00d, "Free Space Wand" },
	{ (0x0d << 16) + 0x020, "Stylus" },
	{ (0x0d << 16) + 0x021, "Puck" },
	{ (0x0d << 16) + 0x022, "Finger" },
	{ (0x0d << 16) + 0x030, "Tip Pressure" },
	{ (0x0d << 16) + 0x031, "Barrel Pressure" },
	{ (0x0d << 16) + 0x032, "In Range" },
	{ (0x0d << 16) + 0x033, "Touch" },
	{ (0x0d << 16) + 0x034, "Untouch" },
	{ (0x0d << 16) + 0x035, "Tap" },
	{ (0x0d << 16) + 0x036, "Quality" },
	{ (0x0d << 16) + 0x037, "Data Valid" },
	{ (0x0d << 16) + 0x038, "Transducer Index" },
	{ (0x0d << 16) + 0x039, "Tablet Function Keys" },
	{ (0x0d << 16) + 0x03a, "Program Change Keys" },
	{ (0x0d << 16) + 0x03b, "Battery Strength" },
	{ (0x0d << 16) + 0x03c, "Invert" },
	{ (0x0d << 16) + 0x03d, "X Tilt" },
	{ (0x0d << 16) + 0x03e, "Y Tilt" },
	{ (0x0d << 16) + 0x03f, "Azimuth" },
	{ (0x0d << 16) + 0x040, "Altitude" },
	{ (0x0d << 16) + 0x041, "Twist" },
	{ (0x0d << 16) + 0x042, "Tip Switch" },
	{ (0x0d << 16) + 0x043, "Secondary Tip Switch" },
	{ (0x0d << 16) + 0x044, "Barrel Switch" },
	{ (0x0d << 16) + 0x045, "Eraser" },
	{ (0x0d << 16) + 0x046, "Tablet Pick" },
	{ (0x0d << 16) + 0x047, "Confidence" },
	{ (0x0d << 16) + 0x048, "Width" },
	{ (0x0d << 16) + 0x049, "Height" },
	{ (0x0d << 16) + 0x051, "Contact ID" },
	{ (0x0d << 16) + 0x052, "Input Mode" },
	{ (0x0d << 16) + 0x053, "Device Index" },
	{ (0x0d << 16) + 0x054, "Contact Count" },
	{ (0x0d << 16) + 0x055, "Maximum Contact Number" },
	{ (0x0f << 16) + 0x000, "Undefined" },
	{ (0x0f << 16) + 0x001, "Physical Interface Device" },
	{ (0x0f << 16) + 0x020, "Normal" },
	{ (0x0f << 16) + 0x021, "Set Effect Report" },
	{ (0x0f << 16) + 0x022, "Effect Block Index" },
	{ (0x0f << 16) + 0x023, "Parameter Block Offset" },
	{ (0x0f << 16) + 0x024, "ROM Flag" },
	{ (0x0f << 16) + 0x025, "Effect Type" },
	{ (0x0f << 16) + 0x026, "ET Constant Force" },
	{ (0x0f << 16) + 0x027, "ET Ramp" },
	{ (0x0f << 16) + 0x028, "ET Custom Force Data" },
	{ (0x0f << 16) + 0x030, "ET Square" },
	{ (0x0f << 16) + 0x031, "ET Sine" },
	{ (0x0f << 16) + 0x032, "ET Triangle" },
	{ (0x0f << 16) + 0x033, "ET Sawtooth Up" },
	{ (0x0f << 16) + 0x034, "ET Sawtooth Down" },
	{ (0x0f << 16) + 0x040, "ET Spring" },
	{ (0x0f << 16) + 0x041, "ET Damper" },
	{ (0x0f << 16) + 0x042, "ET Inertia" },
	{ (0x0f << 16) + 0x043, "ET Friction" },
	{ (0x0f << 16) + 0x050, "Duration" },
	{ (0x0f << 16) + 0x051, "Sample Period" },
	{ (0x0f << 16) + 0x052, "Gain" },
	{ (0x0f << 16) + 0x053, "Trigger Button" },
	{ (0x0f << 16) + 0x054, "Trigger Repeat Interval" },
	{ (0x0f << 16) + 0x055, "Axes Enable" },
	{ (0x0f << 16) + 0x056, "Direction Enable" },
	{ (0x0f << 16) + 0x057, "Direction" },
	{ (0x0f << 16) + 0x058, "Type Specific Block Offset" },
	{ (0x0f << 16) + 0x059, "Block Type" },
	{ (0x0f << 16) + 0x05A, "Set Envelope Report" },
	{ (0x0f << 16) + 0x05B, "Attack Level" },
	{ (0x0f << 16) + 0x05C, "Attack Time" },
	{ (0x0f << 16) + 0x05D, "Fade Level" },
	{ (0x0f << 16) + 0x05E, "Fade Time" },
	{ (0x0f << 16) + 0x05F, "Set Condition Report" },
	{ (0x0f << 16) + 0x060, "CP Offset" },
	{ (0x0f << 16) + 0x061, "Positive Coefficient" },
	{ (0x0f << 16) + 0x062, "Negative Coefficient" },
	{ (0x0f << 16) + 0x063, "Positive Saturation" },
	{ (0x0f << 16) + 0x064, "Negative Saturation" },
	{ (0x0f << 16) + 0x065, "Dead Band" },
	{ (0x0f << 16) + 0x066, "Download Force Sample" },
	{ (0x0f << 16) + 0x067, "Isoch Custom Force Enable" },
	{ (0x0f << 16) + 0x068, "Custom Force Data Report" },
	{ (0x0f << 16) + 0x069, "Custom Force Data" },
	{ (0x0f << 16) + 0x06A, "Custom Force Vendor Defined Data" },
	{ (0x0f << 16) + 0x06B, "Set Custom Force Report" },
	{ (0x0f << 16) + 0x06C, "Custom Force Data Offset" },
	{ (0x0f << 16) + 0x06D, "Sample Count" },
	{ (0x0f << 16) + 0x06E, "Set Periodic Report" },
	{ (0x0f << 16) + 0x06F, "Offset" },
	{ (0x0f << 16) + 0x070, "Magnitude" },
	{ (0x0f << 16) + 0x071, "Phase" },
	{ (0x0f << 16) + 0x072, "Period" },
	{ (0x0f << 16) + 0x073, "Set Constant Force Report" },
	{ (0x0f << 16) + 0x074, "Set Ramp Force Report" },
	{ (0x0f << 16) + 0x075, "Ramp Start" },
	{ (0x0f << 16) + 0x076, "Ramp End" },
	{ (0x0f << 16) + 0x077, "Effect Operation Report" },
	{ (0x0f << 16) + 0x078, "Effect Operation" },
	{ (0x0f << 16) + 0x079, "Op Effect Start" },
	{ (0x0f << 16) + 0x07A, "Op Effect Start Solo" },
	{ (0x0f << 16) + 0x07B, "Op Effect Stop" },
	{ (0x0f << 16) + 0x07C, "Loop Count" },
	{ (0x0f << 16) + 0x07D, "Device Gain Report" },
	{ (0x0f << 16) + 0x07E, "Device Gain" },
	{ (0x0f << 16) + 0x07F, "PID Pool Report" },
	{ (0x0f << 16) + 0x080, "RAM Pool Size" },
	{ (0x0f << 16) + 0x081, "ROM Pool Size" },
	{ (0x0f << 16) + 0x082, "ROM Effect Block Count" },
	{ (0x0f << 16) + 0x083, "Simultaneous Effects Max" },
	{ (0x0f << 16) + 0x084, "Pool Alignment" },
	{ (0x0f << 16) + 0x085, "PID Pool Move Report" },
	{ (0x0f << 16) + 0x086, "Move Source" },
	{ (0x0f << 16) + 0x087, "Move Destination" },
	{ (0x0f << 16) + 0x088, "Move Length" },
	{ (0x0f << 16) + 0x089, "PID Block Load Report" },
	{ (0x0f << 16) + 0x08B, "Block Load Status" },
	{ (0x0f << 16) + 0x08C, "Block Load Success" },
	{ (0x0f << 16) + 0x08D, "Block Load Full" },
	{ (0x0f << 16) + 0x08E, "Block Load Error" },
	{ (0x0f << 16) + 0x08F, "Block Handle" },
	{ (0x0f << 16) + 0x090, "PID Block Free Report" },
	{ (0x0f << 16) + 0x091, "Type Specific Block Handle" },
	{ (0x0f << 16) + 0x092, "PID State Report" },
	{ (0x0f << 16) + 0x094, "Effect Playing" },
	{ (0x0f << 16) + 0x095, "PID Device Control Report" },
	{ (0x0f << 16) + 0x096, "PID Device Control" },
	{ (0x0f << 16) + 0x097, "DC Enable Actuators" },
	{ (0x0f << 16) + 0x098, "DC Disable Actuators" },
	{ (0x0f << 16) + 0x099, "DC Stop All Effects" },
	{ (0x0f << 16) + 0x09A, "DC Device Reset" },
	{ (0x0f << 16) + 0x09B, "DC Device Pause" },
	{ (0x0f << 16) + 0x09C, "DC Device Continue" },
	{ (0x0f << 16) + 0x09F, "Device Paused" },
	{ (0x0f << 16) + 0x0A0, "Actuators Enabled" },
	{ (0x0f << 16) + 0x0A4, "Safety Switch" },
	{ (0x0f << 16) + 0x0A5, "Actuator Override Switch" },
	{ (0x0f << 16) + 0x0A6, "Actuator Power" },
	{ (0x0f << 16) + 0x0A7, "Start Delay" },
	{ (0x0f << 16) + 0x0A8, "Parameter Block Size" },
	{ (0x0f << 16) + 0x0A9, "Device Managed Pool" },
	{ (0x0f << 16) + 0x0AA, "Shared Parameter Blocks" },
	{ (0x0f << 16) + 0x0AB, "Create New Effect Report" },
	{ (0x0f << 16) + 0x0AC, "RAM Pool Available" },
	{ (0x14 << 16) + 0x000, "Undefined" },
	{ (0x14 << 16) + 0x001, "Alphanumeric Display" },
	{ (0x14 << 16) + 0x020, "Display Attributes Report" },
	{ (0x14 << 16) + 0x021, "ASCII Character Set" },
	{ (0x14 << 16) + 0x022, "Data Read Back" },
	{ (0x14 << 16) + 0x023, "Font Read Back" },
	{ (0x14 << 16) + 0x024, "Display Control Report" },
	{ (0x14 << 16) + 0x025, "Clear Display" },
	{ (0x14 << 16) + 0x026, "Display Enable" },
	{ (0x14 << 16) + 0x027, "Screen Saver Delay" },
	{ (0x14 << 16) + 0x028, "Screen Saver Enable" },
	{ (0x14 << 16) + 0x029, "Vertical Scroll" },
	{ (0x14 << 16) + 0x02a, "Horizontal Scroll" },
	{ (0x14 << 16) + 0x02b, "Character Report" },
	{ (0x14 << 16) + 0x02c, "Display Data" },
	{ (0x14 << 16) + 0x02d, "Display Status" },
	{ (0x14 << 16) + 0x02e, "Stat Not Ready" },
	{ (0x14 << 16) + 0x02f, "Stat Ready" },
	{ (0x14 << 16) + 0x030, "Err Not a loadable Character" },
	{ (0x14 << 16) + 0x031, "Err Font Data Cannot Be Read" },
	{ (0x14 << 16) + 0x032, "Cursur Position Report" },
	{ (0x14 << 16) + 0x033, "Row" },
	{ (0x14 << 16) + 0x034, "Column" },
	{ (0x14 << 16) + 0x035, "Rows" },
	{ (0x14 << 16) + 0x036, "Columns" },
	{ (0x14 << 16) + 0x037, "Cursor Pixel Positioning" },
	{ (0x14 << 16) + 0x038, "Cursor Mode" },
	{ (0x14 << 16) + 0x039, "Cursor Enable" },
	{ (0x14 << 16) + 0x03a, "Cursor Blink" },
	{ (0x14 << 16) + 0x03b, "Font Report" },
	{ (0x14 << 16) + 0x03c, "Font Data" },
	{ (0x14 << 16) + 0x03d, "Character Width" },
	{ (0x14 << 16) + 0x03e, "Character Height" },
	{ (0x14 << 16) + 0x03f, "Character Spacing Horizontal" },
	{ (0x14 << 16) + 0x040, "Character Spacing Vertical" },
	{ (0x14 << 16) + 0x041, "Unicode Character Set" },
	{ (0x80 << 16) + 0x001, "Monitor Control" },
	{ (0x80 << 16) + 0x002, "EDID Information" },
	{ (0x80 << 16) + 0x003, "VDIF Information" },
	{ (0x80 << 16) + 0x004, "VESA Version" },
	{ (0x82 << 16) + 0x001, "Degauss" },
	{ (0x82 << 16) + 0x010, "Brightness" },
	{ (0x82 << 16) + 0x012, "Contrast" },
	{ (0x82 << 16) + 0x016, "Red Video Gain" },
	{ (0x82 << 16) + 0x018, "Green Video Gain" },
	{ (0x82 << 16) + 0x01a, "Blue Video Gain" },
	{ (0x82 << 16) + 0x01c, "Focus" },
	{ (0x82 << 16) + 0x020, "Horizontal Position" },
	{ (0x82 << 16) + 0x022, "Horizontal Size" },
	{ (0x82 << 16) + 0x024, "Horizontal Pincushion" },
	{ (0x82 << 16) + 0x026, "Horizontal Pincushion Balance" },
	{ (0x82 << 16) + 0x028, "Horizontal Misconvergence" },
	{ (0x82 << 16) + 0x02a, "Horizontal Linearity" },
	{ (0x82 << 16) + 0x02c, "Horizontal Linearity Balance" },
	{ (0x82 << 16) + 0x030, "Vertical Position" },
	{ (0x82 << 16) + 0x032, "Vertical Size" },
	{ (0x82 << 16) + 0x034, "Vertical Pincushion" },
	{ (0x82 << 16) + 0x036, "Vertical Pincushion Balance" },
	{ (0x82 << 16) + 0x038, "Vertical Misconvergence" },
	{ (0x82 << 16) + 0x03a, "Vertical Linearity" },
	{ (0x82 << 16) + 0x03c, "Vertical Linearity Balance" },
	{ (0x82 << 16) + 0x040, "Parallelogram Balance (Key Distortion)" },
	{ (0x82 << 16) + 0x042, "Trapezoidal Distortion (Key)" },
	{ (0x82 << 16) + 0x044, "Tilt (Rotation)" },
	{ (0x82 << 16) + 0x046, "Top Corner Distortion Control" },
	{ (0x82 << 16) + 0x048, "Top Corner Distortion Balance" },
	{ (0x82 << 16) + 0x04a, "Bottom Corner Distortion Control" },
	{ (0x82 << 16) + 0x04c, "Bottom Corner Distortion Balance" },
	{ (0x82 << 16) + 0x056, "Horizontal Moire" },
	{ (0x82 << 16) + 0x058, "Vertical Moire" },
	{ (0x82 << 16) + 0x05e, "Input Level Select" },
	{ (0x82 << 16) + 0x060, "Input Source Select" },
	{ (0x82 << 16) + 0x06c, "Red Video Black Level" },
	{ (0x82 << 16) + 0x06e, "Green Video Black Level" },
	{ (0x82 << 16) + 0x070, "Blue Video Black Level" },
	{ (0x82 << 16) + 0x0a2, "Auto Size Center" },
	{ (0x82 << 16) + 0x0a4, "Polarity Horizontal Synchronization" },
	{ (0x82 << 16) + 0x0a6, "Polarity Vertical Synchronization" },
	{ (0x82 << 16) + 0x0aa, "Screen Orientation" },
	{ (0x82 << 16) + 0x0ac, "Horizontal Frequency in Hz" },
	{ (0x82 << 16) + 0x0ae, "Vertical Frequency in 0.1 Hz" },
	{ (0x82 << 16) + 0x0b0, "Settings" },
	{ (0x82 << 16) + 0x0ca, "On Screen Display (OSD)" },
	{ (0x82 << 16) + 0x0d4, "Stereo Mode" },
	{ (0x84 << 16) + 0x000, "Undefined" },
	{ (0x84 << 16) + 0x001, "iName" },
	{ (0x84 << 16) + 0x002, "Present Status" },
	{ (0x84 << 16) + 0x003, "Changed Status" },
	{ (0x84 << 16) + 0x004, "UPS" },
	{ (0x84 << 16) + 0x005, "Power Supply" },
	{ (0x84 << 16) + 0x010, "Battery System" },
	{ (0x84 << 16) + 0x011, "Battery System ID" },
	{ (0x84 << 16) + 0x012, "Battery" },
	{ (0x84 << 16) + 0x013, "Battery ID" },
	{ (0x84 << 16) + 0x014, "Charger" },
	{ (0x84 << 16) + 0x015, "Charger ID" },
	{ (0x84 << 16) + 0x016, "Power Converter" },
	{ (0x84 << 16) + 0x017, "Power Converter ID" },
	{ (0x84 << 16) + 0x018, "Outlet System" },
	{ (0x84 << 16) + 0x019, "Outlet System ID" },
	{ (0x84 << 16) + 0x01a, "Input" },
	{ (0x84 << 16) + 0x01b, "Input ID" },
	{ (0x84 << 16) + 0x01c, "Output" },
	{ (0x84 << 16) + 0x01d, "Output ID" },
	{ (0x84 << 16) + 0x01e, "Flow" },
	{ (0x84 << 16) + 0x01f, "Flow ID" },
	{ (0x84 << 16) + 0x020, "Outlet" },
	{ (0x84 << 16) + 0x021, "Outlet ID" },
	{ (0x84 << 16) + 0x022, "Gang" },
	{ (0x84 << 16) + 0x023, "Gang ID" },
	{ (0x84 << 16) + 0x024, "Power Summary" },
	{ (0x84 << 16) + 0x025, "Power Summary ID" },
	{ (0x84 << 16) + 0x030, "Voltage" },
	{ (0x84 << 16) + 0x031, "Current" },
	{ (0x84 << 16) + 0x032, "Frequency" },
	{ (0x84 << 16) + 0x033, "Apparent Power" },
	{ (0x84 << 16) + 0x034, "Active Power" },
	{ (0x84 << 16) + 0x035, "Percent Load" },
	{ (0x84 << 16) + 0x036, "Temperature" },
	{ (0x84 << 16) + 0x037, "Humidity" },
	{ (0x84 << 16) + 0x038, "Bad Count" },
	{ (0x84 << 16) + 0x040, "Config Voltage" },
	{ (0x84 << 16) + 0x041, "Config Current" },
	{ (0x84 << 16) + 0x042, "Config Frequency" },
	{ (0x84 << 16) + 0x043, "Config Apparent Power" },
	{ (0x84 << 16) + 0x044, "Config Active Power" },
	{ (0x84 << 16) + 0x045, "Config Percent Load" },
	{ (0x84 << 16) + 0x046, "Config Temperature" },
	{ (0x84 << 16) + 0x047, "Config Humidity" },
	{ (0x84 << 16) + 0x050, "Switch On Control" },
	{ (0x84 << 16) + 0x051, "Switch Off Control" },
	{ (0x84 << 16) + 0x052, "Toggle Control" },
	{ (0x84 << 16) + 0x053, "Low Voltage Transfer" },
	{ (0x84 << 16) + 0x054, "High Voltage Transfer" },
	{ (0x84 << 16) + 0x055, "Delay Before Reboot" },
	{ (0x84 << 16) + 0x056, "Delay Before Startup" },
	{ (0x84 << 16) + 0x057, "Delay Before Shutdown" },
	{ (0x84 << 16) + 0x058, "Test" },
	{ (0x84 << 16) + 0x059, "Module Reset" },
	{ (0x84 << 16) + 0x05a, "Audible Alarm Control" },
	{ (0x84 << 16) + 0x060, "Present" },
	{ (0x84 << 16) + 0x061, "Good" },
	{ (0x84 << 16) + 0x062, "Internal Failure" },
	{ (0x84 << 16) + 0x063, "Voltage out of range" },
	{ (0x84 << 16) + 0x064, "Frequency out of range" },
	{ (0x84 << 16) + 0x065, "Overload" },
	{ (0x84 << 16) + 0x066, "Over Charged" },
	{ (0x84 << 16) + 0x067, "Over Temperature" },
	{ (0x84 << 16) + 0x068, "Shutdown Requested" },
	{ (0x84 << 16) + 0x069, "Shutdown  Imminent" },
	{ (0x84 << 16) + 0x06a, "Reserved" },
	{ (0x84 << 16) + 0x06b, "Switch On/Off" },
	{ (0x84 << 16) + 0x06c, "Switchable" },
	{ (0x84 << 16) + 0x06d, "Used" },
	{ (0x84 << 16) + 0x06e, "Boost" },
	{ (0x84 << 16) + 0x06f, "Buck" },
	{ (0x84 << 16) + 0x070, "Initialized" },
	{ (0x84 << 16) + 0x071, "Tested" },
	{ (0x84 << 16) + 0x072, "Awaiting Power" },
	{ (0x84 << 16) + 0x073, "Communication Lost" },
	{ (0x84 << 16) + 0x0fd, "iManufacturer" },
	{ (0x84 << 16) + 0x0fe, "iProduct" },
	{ (0x84 << 16) + 0x0ff, "iSerialNumber" },
	{ (0x85 << 16) + 0x000, "Undefined" },
	{ (0x85 << 16) + 0x001, "SMB Battery Mode" },
	{ (0x85 << 16) + 0x002, "SMB Battery Status" },
	{ (0x85 << 16) + 0x003, "SMB Alarm Warning" },
	{ (0x85 << 16) + 0x004, "SMB Charger Mode" },
	{ (0x85 << 16) + 0x005, "SMB Charger Status" },
	{ (0x85 << 16) + 0x006, "SMB Charger Spec Info" },
	{ (0x85 << 16) + 0x007, "SMB Selector State" },
	{ (0x85 << 16) + 0x008, "SMB Selector Presets" },
	{ (0x85 << 16) + 0x009, "SMB Selector Info" },
	{ (0x85 << 16) + 0x010, "Optional Mfg. Function 1" },
	{ (0x85 << 16) + 0x011, "Optional Mfg. Function 2" },
	{ (0x85 << 16) + 0x012, "Optional Mfg. Function 3" },
	{ (0x85 << 16) + 0x013, "Optional Mfg. Function 4" },
	{ (0x85 << 16) + 0x014, "Optional Mfg. Function 5" },
	{ (0x85 << 16) + 0x015, "Connection to SMBus" },
	{ (0x85 << 16) + 0x016, "Output Connection" },
	{ (0x85 << 16) + 0x017, "Charger Connection" },
	{ (0x85 << 16) + 0x018, "Battery Insertion" },
	{ (0x85 << 16) + 0x019, "Use Next" },
	{ (0x85 << 16) + 0x01a, "OK to use" },
	{ (0x85 << 16) + 0x01b, "Battery  Supported" },
	{ (0x85 << 16) + 0x01c, "SelectorRevision" },
	{ (0x85 << 16) + 0x01d, "Charging Indicator" },
	{ (0x85 << 16) + 0x028, "Manufacturer Access" },
	{ (0x85 << 16) + 0x029, "Remaining Capacity Limit" },
	{ (0x85 << 16) + 0x02a, "Remaining Time Limit" },
	{ (0x85 << 16) + 0x02b, "At Rate" },
	{ (0x85 << 16) + 0x02c, "Capacity Mode" },
	{ (0x85 << 16) + 0x02d, "Broadcast To Charger" },
	{ (0x85 << 16) + 0x02e, "Primary Battery" },
	{ (0x85 << 16) + 0x02f, "Charge Controller" },
	{ (0x85 << 16) + 0x040, "Terminate Charge" },
	{ (0x85 << 16) + 0x041, "Terminate Discharge" },
	{ (0x85 << 16) + 0x042, "Below Remaining Capacity Limit" },
	{ (0x85 << 16) + 0x043, "Remaining Time Limit Expired" },
	{ (0x85 << 16) + 0x044, "Charging" },
	{ (0x85 << 16) + 0x045, "Discharging" },
	{ (0x85 << 16) + 0x046, "Fully Charged" },
	{ (0x85 << 16) + 0x047, "Fully Discharged" },
	{ (0x85 << 16) + 0x048, "Conditioning Flag" },
	{ (0x85 << 16) + 0x049, "At Rate OK" },
	{ (0x85 << 16) + 0x04a, "SMB Error Code" },
	{ (0x85 << 16) + 0x04b, "Need Replacement" },
	{ (0x85 << 16) + 0x060, "At Rate Time To Full" },
	{ (0x85 << 16) + 0x061, "At Rate Time To Empty" },
	{ (0x85 << 16) + 0x062, "Average Current" },
	{ (0x85 << 16) + 0x063, "Max Error" },
	{ (0x85 << 16) + 0x064, "Relative State Of Charge" },
	{ (0x85 << 16) + 0x065, "Absolute State Of Charge" },
	{ (0x85 << 16) + 0x066, "Remaining Capacity" },
	{ (0x85 << 16) + 0x067, "Full Charge Capacity" },
	{ (0x85 << 16) + 0x068, "Run Time To Empty" },
	{ (0x85 << 16) + 0x069, "Average Time To Empty" },
	{ (0x85 << 16) + 0x06a, "Average Time To Full" },
	{ (0x85 << 16) + 0x06b, "Cycle Count" },
	{ (0x85 << 16) + 0x080, "Batt. Pack Model Level" },
	{ (0x85 << 16) + 0x081, "Internal Charge Controller" },
	{ (0x85 << 16) + 0x082, "Primary Battery Support" },
	{ (0x85 << 16) + 0x083, "Design Capacity" },
	{ (0x85 << 16) + 0x084, "Specification Info" },
	{ (0x85 << 16) + 0x085, "Manufacturer Date" },
	{ (0x85 << 16) + 0x086, "Serial Number" },
	{ (0x85 << 16) + 0x087, "iManufacturerName" },
	{ (0x85 << 16) + 0x088, "iDeviceName" },
	{ (0x85 << 16) + 0x089, "iDeviceChemistry" },
	{ (0x85 << 16) + 0x08a, "Manufacturer Data" },
	{ (0x85 << 16) + 0x08b, "Rechargeable" },
	{ (0x85 << 16) + 0x08c, "Warning Capacity Limit" },
	{ (0x85 << 16) + 0x08d, "Capacity Granularity 1" },
	{ (0x85 << 16) + 0x08e, "Capacity Granularity 2" },
	{ (0x85 << 16) + 0x08f, "iOEMInformation" },
	{ (0x85 << 16) + 0x0c0, "Inhibit Charge" },
	{ (0x85 << 16) + 0x0c1, "Enable Polling" },
	{ (0x85 << 16) + 0x0c2, "Reset To Zero" },
	{ (0x85 << 16) + 0x0d0, "AC Present" },
	{ (0x85 << 16) + 0x0d1, "Battery Present" },
	{ (0x85 << 16) + 0x0d2, "Power Fail" },
	{ (0x85 << 16) + 0x0d3, "Alarm Inhibited" },
	{ (0x85 << 16) + 0x0d4, "Thermistor Under Range" },
	{ (0x85 << 16) + 0x0d5, "Thermistor Hot" },
	{ (0x85 << 16) + 0x0d6, "Thermistor Cold" },
	{ (0x85 << 16) + 0x0d7, "Thermistor Over Range" },
	{ (0x85 << 16) + 0x0d8, "Voltage Out Of Range" },
	{ (0x85 << 16) + 0x0d9, "Current Out Of Range" },
	{ (0x85 << 16) + 0x0da, "Current Not Regulated" },
	{ (0x85 << 16) + 0x0db, "Voltage Not Regulated" },
	{ (0x85 << 16) + 0x0dc, "Master Mode" },
	{ (0x85 << 16) + 0x0f0, "Charger Selector Support" },
	{ (0x85 << 16) + 0x0f1, "Charger Spec" },
	{ (0x85 << 16) + 0x0f2, "Level 2" },
	{ (0x85 << 16) + 0x0f3, "Level 3" },
	{ (0xf0 << 16) + 0x0f1, "Cash Drawer" },
	{ (0xf0 << 16) + 0x0f2, "Cash Drawer Number" },
	{ (0xf0 << 16) + 0x0f3, "Cash Drawer Set" },
	{ (0xf0 << 16) + 0x0f4, "Cash Drawer Status" },
	{ (0x00 << 16) + 0x000, NULL },
};

/* dialects are represented as:
 *	langid + (dialectid << 10)
 */
static const struct genericstrtable langids[] = {
	{ 0x0001 + (0x01 << 10), "Saudi Arabia" },
	{ 0x0001 + (0x02 << 10), "Iraq" },
	{ 0x0001 + (0x03 << 10), "Egypt" },
	{ 0x0001 + (0x04 << 10), "Libya" },
	{ 0x0001 + (0x05 << 10), "Algeria" },
	{ 0x0001 + (0x06 << 10), "Morocco" },
	{ 0x0001 + (0x07 << 10), "Tunesia" },
	{ 0x0001 + (0x08 << 10), "Oman" },
	{ 0x0001 + (0x09 << 10), "Yemen" },
	{ 0x0001 + (0x0a << 10), "Syria" },
	{ 0x0001 + (0x0b << 10), "Jordan" },
	{ 0x0001 + (0x0c << 10), "Lebanon" },
	{ 0x0001 + (0x0d << 10), "Kuwait" },
	{ 0x0001 + (0x0e << 10), "U.A.E" },
	{ 0x0001 + (0x0f << 10), "Bahrain" },
	{ 0x0001 + (0x10 << 10), "Qatar" },
	{ 0x0001, "Arabic" },
	{ 0x0002, "Bulgarian" },
	{ 0x0003, "Catalan" },
	{ 0x0004 + (0x01 << 10), "Traditional" },
	{ 0x0004 + (0x02 << 10), "Simplified" },
	{ 0x0004 + (0x03 << 10), "Hongkong SAR, PRC" },
	{ 0x0004 + (0x04 << 10), "Singapore" },
	{ 0x0004 + (0x05 << 10), "Macau SAR" },
	{ 0x0004, "Chinese" },
	{ 0x0005, "Czech" },
	{ 0x0006, "Danish" },
	{ 0x0007 + (0x01 << 10), "German" },
	{ 0x0007 + (0x02 << 10), "Swiss" },
	{ 0x0007 + (0x03 << 10), "Austrian" },
	{ 0x0007 + (0x04 << 10), "Luxembourg" },
	{ 0x0007 + (0x05 << 10), "Liechtenstein" },
	{ 0x0007, "German" },
	{ 0x0008, "Greek" },
	{ 0x0009 + (0x01 << 10), "US" },
	{ 0x0009 + (0x02 << 10), "UK" },
	{ 0x0009 + (0x03 << 10), "Australian" },
	{ 0x0009 + (0x04 << 10), "Canadian" },
	{ 0x0009 + (0x05 << 10), "New Zealand" },
	{ 0x0009 + (0x06 << 10), "Ireland" },
	{ 0x0009 + (0x07 << 10), "South Africa" },
	{ 0x0009 + (0x08 << 10), "Jamaica" },
	{ 0x0009 + (0x09 << 10), "Caribbean" },
	{ 0x0009 + (0x0a << 10), "Belize" },
	{ 0x0009 + (0x0b << 10), "Trinidad" },
	{ 0x0009 + (0x0c << 10), "Zimbabwe" },
	{ 0x0009 + (0x0d << 10), "Philippines" },
	{ 0x0009, "English" },
	{ 0x000a + (0x01 << 10), "Castilian" },
	{ 0x000a + (0x02 << 10), "Mexican" },
	{ 0x000a + (0x03 << 10), "Modern" },
	{ 0x000a + (0x04 << 10), "Guatemala" },
	{ 0x000a + (0x05 << 10), "Costa Rica" },
	{ 0x000a + (0x06 << 10), "Panama" },
	{ 0x000a + (0x07 << 10), "Dominican Republic" },
	{ 0x000a + (0x08 << 10), "Venzuela" },
	{ 0x000a + (0x09 << 10), "Colombia" },
	{ 0x000a + (0x0a << 10), "Peru" },
	{ 0x000a + (0x0b << 10), "Argentina" },
	{ 0x000a + (0x0c << 10), "Ecuador" },
	{ 0x000a + (0x0d << 10), "Chile" },
	{ 0x000a + (0x0e << 10), "Uruguay" },
	{ 0x000a + (0x0f << 10), "Paraguay" },
	{ 0x000a + (0x10 << 10), "Bolivia" },
	{ 0x000a + (0x11 << 10), "El Salvador" },
	{ 0x000a + (0x12 << 10), "Honduras" },
	{ 0x000a + (0x13 << 10), "Nicaragua" },
	{ 0x000a + (0x14 << 10), "Puerto Rico" },
	{ 0x000a, "Spanish" },
	{ 0x000b, "Finnish" },
	{ 0x000c + (0x01 << 10), "French" },
	{ 0x000c + (0x02 << 10), "Belgian" },
	{ 0x000c + (0x03 << 10), "Canadian" },
	{ 0x000c + (0x04 << 10), "Swiss" },
	{ 0x000c + (0x05 << 10), "Luxembourg" },
	{ 0x000c + (0x06 << 10), "Monaco" },
	{ 0x000c, "French" },
	{ 0x000d, "Hebrew" },
	{ 0x000e, "Hungarian" },
	{ 0x000f, "Idelandic" },
	{ 0x0010 + (0x01 << 10), "Italian" },
	{ 0x0010 + (0x02 << 10), "Swiss" },
	{ 0x0010, "Italian" },
	{ 0x0011, "Japanese" },
	{ 0x0012 + (0x01 << 10), "Korean" },
	{ 0x0012, "Korean" },
	{ 0x0013 + (0x01 << 10), "Dutch" },
	{ 0x0013 + (0x02 << 10), "Belgian" },
	{ 0x0013, "Dutch" },
	{ 0x0014 + (0x01 << 10), "Bokmal" },
	{ 0x0014 + (0x02 << 10), "Nynorsk" },
	{ 0x0014, "Norwegian" },
	{ 0x0015, "Polish" },
	{ 0x0016 + (0x01 << 10), "Portuguese" },
	{ 0x0016 + (0x02 << 10), "Brazilian" },
	{ 0x0016, "Portuguese" },
	{ 0x0017, "forgotten" },
	{ 0x0018, "Romanian" },
	{ 0x0019, "Russian" },
	{ 0x001a + (0x01 << 10), "Croatian" },
	{ 0x001a + (0x02 << 10), "Latin" },
	{ 0x001a + (0x03 << 10), "Cyrillic" },
	{ 0x001a, "Serbian" },
	{ 0x001b, "Slovak" },
	{ 0x001c, "Albanian" },
	{ 0x001d + (0x01 << 10), "Swedish" },
	{ 0x001d + (0x02 << 10), "Finland" },
	{ 0x001d, "Swedish" },
	{ 0x001e, "Thai" },
	{ 0x001f, "Turkish" },
	{ 0x0020 + (0x01 << 10), "Pakistan" },
	{ 0x0020 + (0x02 << 10), "India" },
	{ 0x0020, "Urdu" },
	{ 0x0021, "Indonesian" },
	{ 0x0022, "Ukrainian" },
	{ 0x0023, "Belarusian" },
	{ 0x0024, "Slovenian" },
	{ 0x0025, "Estonian" },
	{ 0x0026, "Latvian" },
	{ 0x0027 + (0x01 << 10), "Lithuanian" },
	{ 0x0027, "Lithuanian" },
	{ 0x0028, "forgotten" },
	{ 0x0029, "Farsi" },
	{ 0x002a, "Vietnamese" },
	{ 0x002b, "Armenian" },
	{ 0x002c + (0x01 << 10), "Cyrillic" },
	{ 0x002c + (0x02 << 10), "Latin" },
	{ 0x002c, "Azeri" },
	{ 0x002d, "Basque" },
	{ 0x002e, "forgotten" },
	{ 0x002f, "Macedonian" },
	{ 0x0036, "Afrikaans" },
	{ 0x0037, "Georgian" },
	{ 0x0038, "Faeroese" },
	{ 0x0039, "Hindi" },
	{ 0x003e + (0x01 << 10), "Malaysia" },
	{ 0x003e + (0x02 << 10), "Brunei Darassalam" },
	{ 0x003e, "Malay" },
	{ 0x003f, "Kazak" },
	{ 0x0041, "Awahili" },
	{ 0x0043 + (0x01 << 10), "Latin" },
	{ 0x0043 + (0x02 << 10), "Cyrillic" },
	{ 0x0043, "Uzbek" },
	{ 0x0044, "Tatar" },
	{ 0x0045, "Bengali" },
	{ 0x0046, "Punjabi" },
	{ 0x0047, "Gujarati" },
	{ 0x0048, "Oriya" },
	{ 0x0049, "Tamil" },
	{ 0x004a, "Telugu" },
	{ 0x004b, "Kannada" },
	{ 0x004c, "Malayalam" },
	{ 0x004d, "Assamese" },
	{ 0x004e, "Marathi" },
	{ 0x004f, "Sanskrit" },
	{ 0x0057, "Konkani" },
	{ 0x0058, "Manipuri" },
	{ 0x0059, "Sindhi" },
	{ 0x0060 + (0x02 << 10), "India" },
	{ 0x0060, "Kashmiri" },
	{ 0x0061 + (0x02 << 10), "India" },
	{ 0x0061, "Nepali" },
	{ 0x0000, NULL },
};

static const struct genericstrtable countrycodes[] = {
	{ 0, "Not supported" },
	{ 1, "Arabic" },
	{ 2, "Belgian" },
	{ 3, "Canadian-Bilingual" },
	{ 4, "Canadian-French" },
	{ 5, "Czech Republic" },
	{ 6, "Danish" },
	{ 7, "Finnish" },
	{ 8, "French" },
	{ 9, "German" },
	{ 10, "Greek" },
	{ 11, "Hebrew" },
	{ 12, "Hungary" },
	{ 13, "International (ISO)" },
	{ 14, "Italian" },
	{ 15, "Japan (Katakana)" },
	{ 16, "Korean" },
	{ 17, "Latin American" },
	{ 18, "Netherlands/Dutch" },
	{ 19, "Norwegian" },
	{ 20, "Persian (Farsi)" },
	{ 21, "Poland" },
	{ 22, "Portuguese" },
	{ 23, "Russia" },
	{ 24, "Slovakia" },
	{ 25, "Spanish" },
	{ 26, "Swedish" },
	{ 27, "Swiss/French" },
	{ 28, "Swiss/German" },
	{ 29, "Switzerland" },
	{ 30, "Taiwan" },
	{ 31, "Turkish-Q" },
	{ 32, "UK" },
	{ 33, "US" },
	{ 34, "Yugoslavia" },
	{ 35, "Turkish-F" },
	{ 0, NULL },
};

/* ---------------------------------------------------------------------- */
#endif /* _USB_SPEC_H */
