/*
 * Copyright (C) 2010 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ScopedEventQueue_h
#define ScopedEventQueue_h

#include <wtf/NeverDestroyed.h>
#include <wtf/Noncopyable.h>
#include <wtf/PassRefPtr.h>
#include <wtf/RefPtr.h>
#include <wtf/Vector.h>

namespace WebCore {

class Event;
class EventQueueScope;

class ScopedEventQueue {
    WTF_MAKE_NONCOPYABLE(ScopedEventQueue); WTF_MAKE_FAST_ALLOCATED;
public:
    static ScopedEventQueue& singleton();
    void enqueueEvent(PassRefPtr<Event>);

private:
    ScopedEventQueue() = default;
    ~ScopedEventQueue() = delete;

    void dispatchEvent(PassRefPtr<Event>) const;
    void dispatchAllEvents();
    void incrementScopingLevel();
    void decrementScopingLevel();

    Vector<RefPtr<Event>> m_queuedEvents;
    unsigned m_scopingLevel { 0 };

    friend class WTF::NeverDestroyed<WebCore::ScopedEventQueue>;
    friend class EventQueueScope;
};

class EventQueueScope {
    WTF_MAKE_NONCOPYABLE(EventQueueScope);
public:
    EventQueueScope() { ScopedEventQueue::singleton().incrementScopingLevel(); }
    ~EventQueueScope() { ScopedEventQueue::singleton().decrementScopingLevel(); }
};

}

#endif // ScopedEventQueue_h
