(* ========================================================================= *)
(* edwards25519, an Edwards-coordinate version of curve25519.                *)
(* ========================================================================= *)

needs "EC/edwards.ml";;
needs "EC/excluderoots.ml";;
needs "EC/computegroup.ml";;

add_curve edwards_curve;;
add_curvezero edwards_0;;
add_curveneg edwards_neg;;
add_curveadd edwards_add;;

(* ------------------------------------------------------------------------- *)
(* Parameters for the edwards25519 curve, mainly from the document           *)
(* https://datatracker.ietf.org/doc/draft-ietf-lwig-curve-representations/   *)
(* Here  n_25519 is the large prime factor of the group order, the full      *)
(* group order being 8 * n_25519. Likewise E_25519 is the generator of the   *)
(* prime order subgroup and EE_25519 is a generator for the full group.      *)
(* We use e_25519 instead of a_25519 as the name, as the latter is also used *)
(* for the Weierstrass form parameter in the wei25519.ml file.               *)
(* ------------------------------------------------------------------------- *)

let p_25519 = define`p_25519 = 57896044618658097711785492504343953926634992332820282019728792003956564819949`;;
let n_25519 = define`n_25519 = 7237005577332262213973186563042994240857116359379907606001950938285454250989`;;
let e_25519 = define`e_25519 = 57896044618658097711785492504343953926634992332820282019728792003956564819948`;;
let d_25519 = define`d_25519 = 37095705934669439343138083508754565189542113879843219016388785533085940283555`;;
let E_25519 = define `E_25519 = (&0x216936d3cd6e53fec0a4e231fdd6dc5c692cc7609525a7b2c9562d608f25d51a:int,&0x6666666666666666666666666666666666666666666666666666666666666658:int)`;;
let EE_25519 = define `EE_25519 = (&31673755162551823185009131889882229316567966938545344397180697279978689160235:int, &44332021396607921014542692927380285321101622747762054416338596910403709737434:int)`;;

let edwards25519 = define
 `edwards25519 = (integer_mod_ring p_25519,&e_25519:int,&d_25519:int)`;;

(* ------------------------------------------------------------------------- *)
(* Primality of the field characteristic and (sub)group order.               *)
(* ------------------------------------------------------------------------- *)

let P_25519 = prove
 (`p_25519 = 2 EXP 255 - 19`,
  REWRITE_TAC[p_25519] THEN ARITH_TAC);;

let N_25519 = prove
 (`n_25519 = 2 EXP 252 + 27742317777372353535851937790883648493`,
  REWRITE_TAC[n_25519] THEN ARITH_TAC);;

let PRIME_P25519 = prove
 (`prime p_25519`,
  REWRITE_TAC[p_25519] THEN (CONV_TAC o PRIME_RULE)
   ["2"; "3"; "5"; "7"; "11"; "13"; "17"; "19"; "23"; "29"; "31"; "37"; "41";
    "43"; "47"; "53"; "59"; "83"; "97"; "103"; "107"; "127"; "131"; "173";
    "223"; "239"; "353"; "419"; "479"; "487"; "991"; "1723"; "2437"; "3727";
    "4153"; "9463"; "32573"; "37853"; "57467"; "65147"; "75707"; "132049";
    "430751"; "569003"; "1923133"; "8574133"; "2773320623"; "72106336199";
    "1919519569386763"; "31757755568855353";
    "75445702479781427272750846543864801";
    "74058212732561358302231226437062788676166966415465897661863160754340907";
    "57896044618658097711785492504343953926634992332820282019728792003956564819949"]);;

let PRIME_N25519 = prove
 (`prime n_25519`,
  REWRITE_TAC[n_25519] THEN (CONV_TAC o PRIME_RULE)
  ["2"; "3"; "5"; "7"; "11"; "13"; "17"; "19"; "23"; "41"; "43"; "47"; "67";
   "73"; "79"; "113"; "269"; "307"; "1361"; "1723"; "2551"; "2851"; "2939";
   "3797"; "5879"; "17231"; "22111"; "30703"; "34123"; "41081"; "82163";
   "132667"; "137849"; "409477"; "531581"; "1224481"; "14741173"; "58964693";
   "292386187"; "213441916511"; "1257559732178653"; "4434155615661930479";
   "3044861653679985063343"; "172054593956031949258510691";
   "198211423230930754013084525763697";
   "19757330305831588566944191468367130476339";
   "276602624281642239937218680557139826668747";
   "7237005577332262213973186563042994240857116359379907606001950938285454250989"]);;

(* ------------------------------------------------------------------------- *)
(* Definition of the curve group and proof of its key properties.            *)
(* ------------------------------------------------------------------------- *)

let edwards25519_group = define
 `edwards25519_group =
    edwards_group(integer_mod_ring p_25519,&e_25519,&d_25519)`;;

let EDWARDS_NONSINGULAR_25519 = prove
 (`edwards_nonsingular (integer_mod_ring p_25519,&e_25519,&d_25519)`,
  REWRITE_TAC[edwards_nonsingular; INTEGER_MOD_RING_ROOT_EXISTS] THEN
  SIMP_TAC[INTEGER_MOD_RING; INT_OF_NUM_EQ; e_25519; d_25519; p_25519] THEN
  CONV_TAC NUM_REDUCE_CONV THEN
  SIMP_TAC[EULER_CRITERION; REWRITE_RULE[p_25519] PRIME_P25519] THEN
  CONV_TAC(DEPTH_CONV
   (NUM_SUB_CONV ORELSEC NUM_DIV_CONV ORELSEC DIVIDES_CONV)) THEN
  REWRITE_TAC[CONG] THEN CONV_TAC(ONCE_DEPTH_CONV EXP_MOD_CONV) THEN
  CONV_TAC NUM_REDUCE_CONV);;

let EDWARDS25519_GROUP = prove
 (`group_carrier edwards25519_group =
     edwards_curve(integer_mod_ring p_25519,&e_25519,&d_25519) /\
   group_id edwards25519_group =
     edwards_0(integer_mod_ring p_25519,&e_25519,&d_25519) /\
   group_inv edwards25519_group =
     edwards_neg(integer_mod_ring p_25519,&e_25519,&d_25519) /\
   group_mul edwards25519_group =
     edwards_add(integer_mod_ring p_25519,&e_25519,&d_25519)`,
  REWRITE_TAC[edwards25519_group] THEN MATCH_MP_TAC EDWARDS_GROUP THEN
  REWRITE_TAC[EDWARDS_NONSINGULAR_25519] THEN
  REWRITE_TAC[FIELD_INTEGER_MOD_RING; PRIME_P25519] THEN
  REWRITE_TAC[e_25519; d_25519; p_25519] THEN
  REWRITE_TAC[IN_INTEGER_MOD_RING_CARRIER] THEN CONV_TAC INT_REDUCE_CONV);;

add_ecgroup [e_25519; d_25519; p_25519] EDWARDS25519_GROUP;;

let FINITE_GROUP_CARRIER_EDWARDS25519 = prove
 (`FINITE(group_carrier edwards25519_group)`,
  REWRITE_TAC[EDWARDS25519_GROUP] THEN MATCH_MP_TAC FINITE_EDWARDS_CURVE THEN
  REWRITE_TAC[FINITE_INTEGER_MOD_RING;
              FIELD_INTEGER_MOD_RING; PRIME_P25519] THEN
  REWRITE_TAC[p_25519] THEN CONV_TAC NUM_REDUCE_CONV);;

let GENERATOR_IN_GROUP_CARRIER_EDWARDS25519 = prove
 (`E_25519 IN group_carrier edwards25519_group`,
  REWRITE_TAC[E_25519] THEN CONV_TAC ECGROUP_CARRIER_CONV);;

let GROUP_ELEMENT_ORDER_EDWARDS25519_E25519 = prove
 (`group_element_order edwards25519_group E_25519 = n_25519`,
  SIMP_TAC[GROUP_ELEMENT_ORDER_UNIQUE_PRIME;
           GENERATOR_IN_GROUP_CARRIER_EDWARDS25519; PRIME_N25519] THEN
  REWRITE_TAC[E_25519; el 1 (CONJUNCTS EDWARDS25519_GROUP)] THEN
  REWRITE_TAC[edwards_0; PAIR_EQ; INTEGER_MOD_RING] THEN
  REWRITE_TAC[n_25519; p_25519] THEN CONV_TAC INT_REDUCE_CONV THEN
  CONV_TAC(LAND_CONV ECGROUP_POW_CONV) THEN REFL_TAC);;

let FULLGENERATOR_IN_GROUP_CARRIER_EDWARDS25519 = prove
 (`EE_25519 IN group_carrier edwards25519_group`,
  REWRITE_TAC[EE_25519] THEN CONV_TAC ECGROUP_CARRIER_CONV);;

let GROUP_ELEMENT_ORDER_EDWARDS25519_EE25519 = prove
 (`group_element_order edwards25519_group EE_25519 = 8 * n_25519`,
  ABBREV_TAC
   `h = (&14399317868200118260347934320527232580618823971194345261214217575416788799818,
         &2707385501144840649318225287225658788936804267575313519463743609750303402022)
    :(int#int)` THEN
  SUBGOAL_THEN
   `h IN group_carrier edwards25519_group /\
    group_element_order edwards25519_group h = 8`
  STRIP_ASSUME_TAC THENL
   [EXPAND_TAC "h" THEN
    MATCH_MP_TAC(TAUT `p /\ (p ==> q) ==> p /\ q`) THEN CONJ_TAC THENL
     [CONV_TAC ECGROUP_CARRIER_CONV;
      SIMP_TAC[GROUP_ELEMENT_ORDER_UNIQUE_ALT; ARITH]] THEN
    DISCH_TAC THEN REWRITE_TAC[EDWARDS25519_GROUP; edwards_0; INTEGER_MOD_RING] THEN
    CONJ_TAC THENL
     [CONV_TAC(LAND_CONV ECGROUP_POW_CONV) THEN
      REWRITE_TAC[p_25519] THEN CONV_TAC INT_REDUCE_CONV;
      ALL_TAC] THEN
    REWRITE_TAC[IMP_CONJ_ALT] THEN CONV_TAC EXPAND_CASES_CONV THEN
    CONV_TAC NUM_REDUCE_CONV THEN REPEAT CONJ_TAC THEN
    CONV_TAC(RAND_CONV(LAND_CONV ECGROUP_POW_CONV)) THEN
    REWRITE_TAC[PAIR_EQ; p_25519] THEN CONV_TAC INT_REDUCE_CONV;
    ALL_TAC] THEN
  SUBGOAL_THEN `EE_25519 = group_mul edwards25519_group h E_25519` SUBST1_TAC THENL
   [EXPAND_TAC "h" THEN REWRITE_TAC[E_25519; EE_25519] THEN
    CONV_TAC(RAND_CONV ECGROUP_MUL_CONV) THEN REFL_TAC;
    ALL_TAC] THEN
  W(MP_TAC o PART_MATCH (lhand o rand) GROUP_ELEMENT_ORDER_MUL_EQ o
    lhand o snd) THEN
  ASM_REWRITE_TAC[GROUP_ELEMENT_ORDER_EDWARDS25519_E25519] THEN
  DISCH_THEN MATCH_MP_TAC THEN
  REWRITE_TAC[GENERATOR_IN_GROUP_CARRIER_EDWARDS25519] THEN CONJ_TAC THENL
   [EXPAND_TAC "h" THEN REWRITE_TAC[E_25519] THEN
    CONV_TAC(BINOP_CONV ECGROUP_MUL_CONV) THEN REFL_TAC;
    REWRITE_TAC[n_25519] THEN CONV_TAC COPRIME_CONV]);;

let SIZE_EDWARDS25519_GROUP = prove
 (`group_carrier edwards25519_group HAS_SIZE (8 * n_25519)`,
  REWRITE_TAC[HAS_SIZE; FINITE_GROUP_CARRIER_EDWARDS25519] THEN
  MP_TAC(ISPECL [`edwards25519_group`; `EE_25519`]
    GROUP_ELEMENT_ORDER_DIVIDES_GROUP_ORDER) THEN
  REWRITE_TAC[FINITE_GROUP_CARRIER_EDWARDS25519;
              FULLGENERATOR_IN_GROUP_CARRIER_EDWARDS25519] THEN
  REWRITE_TAC[divides; LEFT_IMP_EXISTS_THM;
               GROUP_ELEMENT_ORDER_EDWARDS25519_EE25519] THEN
  X_GEN_TAC `d:num` THEN REPEAT_TCL DISJ_CASES_THEN ASSUME_TAC
   (ARITH_RULE `d = 0 \/ d = 1 \/ 2 <= d`) THEN
  ASM_SIMP_TAC[CARD_EQ_0; FINITE_GROUP_CARRIER_EDWARDS25519;
               MULT_CLAUSES; GROUP_CARRIER_NONEMPTY] THEN
  MATCH_MP_TAC(ARITH_RULE
   `s < 16 * n /\ 2 * n <= d * n ==> s = (8 * n) * d ==> x = 8 * n`) THEN
  REWRITE_TAC[LE_MULT_RCANCEL; n_25519; ARITH_EQ] THEN
  ASM_REWRITE_TAC[GSYM n_25519; EDWARDS25519_GROUP] THEN
  W(MP_TAC o PART_MATCH (lhand o rand) CARD_BOUND_EDWARDS_CURVE o
    lhand o snd) THEN
  REWRITE_TAC[EDWARDS_NONSINGULAR_25519] THEN
  REWRITE_TAC[FIELD_INTEGER_MOD_RING; PRIME_P25519] THEN
   SIMP_TAC[FINITE_INTEGER_MOD_RING; CARD_INTEGER_MOD_RING;
           IN_INTEGER_MOD_RING_CARRIER; n_25519; p_25519; d_25519;
           INTEGER_MOD_RING; e_25519; INT_OF_NUM_CLAUSES; ARITH_EQ] THEN
  ARITH_TAC);;

let GENERATED_EDWARDS25519_GROUP = prove
 (`subgroup_generated edwards25519_group {EE_25519} = edwards25519_group`,
  SIMP_TAC[SUBGROUP_GENERATED_ELEMENT_ORDER;
           FULLGENERATOR_IN_GROUP_CARRIER_EDWARDS25519;
           FINITE_GROUP_CARRIER_EDWARDS25519] THEN
  REWRITE_TAC[GROUP_ELEMENT_ORDER_EDWARDS25519_EE25519;
              REWRITE_RULE[HAS_SIZE] SIZE_EDWARDS25519_GROUP]);;

let CYCLIC_EDWARDS25519_GROUP = prove
 (`cyclic_group edwards25519_group`,
  MESON_TAC[CYCLIC_GROUP_ALT; GENERATED_EDWARDS25519_GROUP]);;

let ABELIAN_EDWARDS25519_GROUP = prove
 (`abelian_group edwards25519_group`,
  MESON_TAC[CYCLIC_EDWARDS25519_GROUP; CYCLIC_IMP_ABELIAN_GROUP]);;
