/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isNonNegativeInteger = require( './../../../../base/assert/is-nonnegative-integer' );
var isnan = require( './../../../../base/assert/is-nan' );
var isOdd = require( './../../../../base/assert/is-odd' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var BERNOULLI = require( './bernoulli.json' );


// VARIABLES //

var MAX_BERNOULLI = 258|0; // asm type annotation


// MAIN //

/**
* Computes the nth Bernoulli number.
*
* @param {NonNegativeInteger} n - the Bernoulli number to compute
* @returns {number} Bernoulli number
*
* @example
* var y = bernoulli( 0 );
* // returns 1.0
*
* @example
* var y = bernoulli( 1 );
* // returns 0.0
*
* @example
* var y = bernoulli( 2 );
* // returns ~0.167
*
* @example
* var y = bernoulli( 3 );
* // returns 0.0
*
* @example
* var y = bernoulli( 4 );
* // returns ~-0.033
*
* @example
* var y = bernoulli( 5 );
* // returns 0.0
*
* @example
* var y = bernoulli( 20 );
* // returns ~-529.124
*
* @example
* var y = bernoulli( 260 );
* // returns -Infinity
*
* @example
* var y = bernoulli( 262 );
* // returns Infinity
*
* @example
* var y = bernoulli( NaN );
* // returns NaN
*/
function bernoulli( n ) {
	if ( isnan( n ) || !isNonNegativeInteger( n ) ) {
		return NaN;
	}
	if ( isOdd( n ) ) {
		return 0.0;
	}
	if ( n > MAX_BERNOULLI ) {
		return ( (n/2)&1 ) ? PINF : NINF;
	}
	return BERNOULLI[ n/2 ];
}


// EXPORTS //

module.exports = bernoulli;
