/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _BINARY_EXPRESSION_H_
#define _BINARY_EXPRESSION_H_

#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/AccessorExpression.h"

namespace GTLCore {

  namespace AST {
    class AccessorExpression;
    /**
     * @internal
     * This is the base class of binary expressions ( '<', '==', '+', '%'...).
     * @ingroup GTLCore_AST
     */
    class GTLCORE_EXPORT BinaryExpression : public Expression {
      public:
        BinaryExpression( Expression* lhs, Expression* rhs) : m_lhs(lhs), m_rhs(rhs) {}
        ~BinaryExpression();
        int priority();
        Expression* leftHandSide() { return m_lhs; }
        const Expression* leftHandSide() const { return m_lhs; }
        Expression* rightHandSide() { return m_rhs; }
        const Expression* rightHandSide() const { return m_rhs; }
        virtual const GTLCore::Type* type() const;
        virtual bool isConstant() const
        {
          return m_lhs->isConstant() or m_rhs->isConstant();
        }
        virtual void markAsReturnExpression();
      private:
        Expression* m_lhs;
        Expression* m_rhs;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class OrBinaryExpression : public BinaryExpression {
      public:
        OrBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class AndBinaryExpression : public BinaryExpression {
      public:
        AndBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class BitOrBinaryExpression : public BinaryExpression {
      public:
        BitOrBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class BitXorBinaryExpression : public BinaryExpression {
      public:
        BitXorBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class BitAndBinaryExpression : public BinaryExpression {
      public:
        BitAndBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class EqualEqualBinaryExpression : public BinaryExpression {
      public:
        EqualEqualBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class DifferentBinaryExpression : public BinaryExpression {
      public:
        DifferentBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual const GTLCore::Type* type() const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class InferiorEqualBinaryExpression : public BinaryExpression {
      public:
        InferiorEqualBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class InferiorBinaryExpression : public BinaryExpression {
      public:
        InferiorBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class SupperiorEqualBinaryExpression : public BinaryExpression {
      public:
        SupperiorEqualBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class SupperiorBinaryExpression : public BinaryExpression {
      public:
        SupperiorBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
        virtual const GTLCore::Type* type() const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class RightShiftBinaryExpression : public BinaryExpression {
      public:
        RightShiftBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class LeftShiftBinaryExpression : public BinaryExpression {
      public:
        LeftShiftBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class GTLCORE_TESTS_EXPORT AdditionBinaryExpression : public BinaryExpression {
      public:
        AdditionBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class SubtractionBinaryExpression : public BinaryExpression {
      public:
        SubtractionBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class MultiplicationBinaryExpression : public BinaryExpression {
      public:
        MultiplicationBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class GTLCORE_TESTS_EXPORT DivisionBinaryExpression : public BinaryExpression {
      public:
        DivisionBinaryExpression ( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class ModuloBinaryExpression : public BinaryExpression {
      public:
        ModuloBinaryExpression( Expression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs )
        {
        }
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
    };
    /**
     * @internal
     * This is the class representing an assignement of a value to a variable, the left hand side of this
     * expression must be a @ref VariableExpression.
     * @ingroup GTLCore_AST
     */
    class GTLCORE_EXPORT AssignementBinaryExpression : public BinaryExpression {
      public:
        AssignementBinaryExpression( AccessorExpression* lhs, Expression* rhs );
        virtual llvm::BasicBlock* generateStatement( LLVMBackend::GenerationContext&, llvm::BasicBlock* ) const;
        virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const;
        virtual ExpressionResultSP generateValue( GenerationVisitor* _generationVisitor) const;
      private:
        AccessorExpression* m_lhs;
    };
  }
}

#endif
