///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file StandardConstControllers.h 
 * \brief Contains the definition of the Core::StandardConstController class and 
 *        its derived classes. 
 */

#ifndef __OVITO_STD_CONST_CONTROLLERS_H
#define __OVITO_STD_CONST_CONTROLLERS_H

#include <core/Core.h>
#include "Controller.h"
#include <core/undo/UndoManager.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

template<class BaseControllerClass, typename ValueType, typename NullValue, class AddFunction = plus<ValueType> >
class StandardConstController : public BaseControllerClass 
{
	typedef StandardConstController<BaseControllerClass, ValueType, NullValue, AddFunction> ThisClassType;
	
protected:
	// Define an undo class that can restore the old value.
	class ChangeValueOperation : public UndoableOperation {
	public:
		ChangeValueOperation(StandardConstController* ctrl) : controller(ctrl), storedValue(ctrl->value) {}
		virtual void undo() {
			// Restore old controller value.
			swap(controller->value, storedValue);
			controller->notifyDependents(REFTARGET_CHANGED);
		}
		virtual void redo() { undo(); }
		/// Provides a localized, human readable description of this operation.
		virtual QString displayName() const { return "Constant controller change"; }
	private:
		intrusive_ptr<StandardConstController> controller;
		ValueType storedValue;
	};

public:
    // Default constructor.
	StandardConstController(bool isLoading) : BaseControllerClass(isLoading), value(NullValue()) {}

	/// Queries the controller for its absolute value.
	virtual void getValue(TimeTicks time, ValueType& result, TimeInterval& validityInterval) {
		result = value;
	}

	/// Sets the controller's value at the specified time.
	virtual void setValue(TimeTicks time, const ValueType& _newValue, bool isAbsoluteValue) {
		ValueType newValue;
		if(!isAbsoluteValue) {
			AddFunction func;
			newValue = func(_newValue, value);
		}
		else newValue = _newValue;
		if(newValue == value) return;	// No value change.

		// Handle undo
		if(UNDO_MANAGER.isRecording()) 
			UNDO_MANAGER.addOperation(new ChangeValueOperation(this));

		value = newValue;

		// Send change message.
		this->notifyDependents(REFTARGET_CHANGED);
	}

protected:

	/// Saves the class' contents to the given stream. 
	virtual void saveToStream(ObjectSaveStream& stream) {
		BaseControllerClass::saveToStream(stream);
		stream.beginChunk(0xA499BC);
		stream << value;
		stream.endChunk(); 
	}

	/// Loads the class' contents from the given stream. 
	virtual void loadFromStream(ObjectLoadStream& stream) {
		BaseControllerClass::loadFromStream(stream);
		stream.expectChunk(0xA499BC);
		stream >> value;
		stream.closeChunk(); 
	}

	/// Creates a copy of this object. 
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper) {
		// Let the base class create an instance of this class.
		intrusive_ptr<ThisClassType> clone = static_object_cast<ThisClassType>(BaseControllerClass::clone(deepCopy, cloneHelper));
		clone->value = this->value;
		return clone;
	}

	/// The value of the constant controller.
    ValueType value;
};

// Define some standard constant controllers.
class CORE_DLLEXPORT ConstFloatController : public StandardConstController<FloatController, FloatType, FloatType> {
public:
	ConstFloatController(bool isLoading = false) : StandardConstController<FloatController, FloatType, FloatType>(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ConstFloatController)
};
class CORE_DLLEXPORT ConstIntegerController : public StandardConstController<IntegerController, int, int> {
public:
	ConstIntegerController(bool isLoading = false) : StandardConstController<IntegerController, int, int>(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ConstIntegerController)
};
class CORE_DLLEXPORT ConstVectorController : public StandardConstController<VectorController, Vector3, NullVector> {
public:
	ConstVectorController(bool isLoading = false) : StandardConstController<VectorController, Vector3, NullVector>(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ConstVectorController)
};

struct _BooleanValueAddFunction : public binary_function<bool, bool, bool> {
	bool operator()(const bool& b1, const bool& b2) const { return b2; }
};
class CORE_DLLEXPORT ConstBooleanController : public StandardConstController<BooleanController, bool, bool, _BooleanValueAddFunction> {
public:
	ConstBooleanController(bool isLoading = false) : StandardConstController<BooleanController, bool, bool, _BooleanValueAddFunction>(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ConstBooleanController)
};

};

#endif // __OVITO_STD_CONST_CONTROLLERS_H
